import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import {
  ChildPaginationDto,
  CreateByUserChildDto,
  CreateUserChildDto,
  CSVChildPaginationDto,
  SearchChildDto,
} from './dto/create-user-child.dto';
import { UpdateUserChildDto } from './dto/update-user-child.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { UserChild } from './entities/user-child.entity';
import { Brackets, Not, Repository } from 'typeorm';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { DefaultStatus, MemberStatus, UserRole } from 'src/enum';
import { MembershipCard } from 'src/membership-card/entities/membership-card.entity';
import { generateQrCode } from 'src/utils/qrCode.util';
import { unlink } from 'fs/promises';
import { join } from 'path';
import { createObjectCsvStringifier } from 'csv-writer';

@Injectable()
export class UserChildService {
  constructor(
    @InjectRepository(UserChild) private readonly repo: Repository<UserChild>,
    @InjectRepository(MembershipCard)
    private readonly memCardRepo: Repository<MembershipCard>,
    @InjectRepository(UserDetail)
    private readonly udRepo: Repository<UserDetail>,
  ) {}

  async create(dto: CreateUserChildDto) {
    const user = await this.udRepo.findOne({
      where: { accountId: dto.accountId },
    });
    if (!user) {
      throw new NotFoundException('User Not Found!');
    }
    const card = await this.memCardRepo.findOne({
      where: { id: user.membershipCardId },
    });
    if (!card) {
      throw new NotFoundException('MembershipCard Not found');
    }
    const memberCount = card.memberCount;
    const childCount = await this.repo.count({
      where: { accountId: dto.accountId },
    });
    if (childCount == memberCount) {
      throw new ConflictException('Cannot add child member, limit exceeded!');
    }
    if (user.status != MemberStatus.ACTIVE) {
      throw new ConflictException('Member is not acitve');
    }
    const child = await this.repo.findOne({
      where: { accountId: dto.accountId, name: dto.name },
    });
    if (child) {
      throw new ConflictException(
        'Child Member already exists with this name!',
      );
    }
    if (dto.relation === 'Spouse') {
      const parentMemId = user.memberId.replace(/-M$/, '-S');
      dto.memberId = parentMemId;
    } else {
      const latestDependent = await this.repo.findOne({
        where: { accountId: user.accountId, relation: Not('Spouse') },
        order: { createAt: 'DESC' },
      });
      const capitalLetters = [
        'A',
        'B',
        'C',
        'D',
        'E',
        'F',
        'G',
        'H',
        'I',
        'J',
        'K',
        'L',
        'M',
        'N',
        'O',
        'P',
        'Q',
        'R',
        'T',
        'U',
        'V',
        'W',
        'X',
        'Y',
        'Z',
      ];
      let nextChar = 'A';
      if (latestDependent) {
        const lastChar = latestDependent.memberId.slice(-1);
        const currentIndex = capitalLetters.indexOf(lastChar);
        nextChar = capitalLetters[currentIndex + 1] || 'A';
      }
      const parentMemId = user.memberId.replace(/-M$/, `-${nextChar}`);
      dto.memberId = parentMemId;
    }
    const obj = Object.assign(dto);
    return this.repo.save(obj);
  }

  async addChild(dto: CreateByUserChildDto) {
    const user = await this.udRepo.findOne({
      where: { accountId: dto.accountId },
    });
    if (!user) {
      throw new NotFoundException('User Not Found!');
    }
    const card = await this.memCardRepo.findOne({
      where: { id: user.membershipCardId },
    });
    if (!card) {
      throw new NotFoundException('MembershipCard Not found');
    }
    const memberCount = card.memberCount;
    const childCount = await this.repo.count({
      where: { accountId: dto.accountId },
    });
    if (childCount == memberCount) {
      throw new ConflictException('Cannot add child member, limit exceeded!');
    }
    if (user.status != MemberStatus.ACTIVE) {
      throw new ConflictException('Member is not acitve');
    }
    const child = await this.repo.findOne({
      where: { accountId: dto.accountId, name: dto.name },
    });
    if (child) {
      throw new ConflictException(
        'Child Member already exists with this name!',
      );
    }

    if (dto.relation === 'Spouse') {
      const parentMemId = user.memberId.replace(/-M$/, '-S');
      dto.memberId = parentMemId;
    } else {
      const latestDependent = await this.repo.findOne({
        where: { accountId: user.accountId, relation: Not('Spouse') },
        order: { createAt: 'DESC' },
      });

      const capitalLetters = [
        'A',
        'B',
        'C',
        'D',
        'E',
        'F',
        'G',
        'H',
        'I',
        'J',
        'K',
        'L',
        'M',
        'N',
        'O',
        'P',
        'Q',
        'R',
        'T',
        'U',
        'V',
        'W',
        'X',
        'Y',
        'Z',
      ];

      let nextChar = 'A';

      if (latestDependent) {
        const lastChar = latestDependent.memberId.slice(-1);
        const currentIndex = capitalLetters.indexOf(lastChar);
        nextChar = capitalLetters[currentIndex + 1] || 'A';
      }

      const parentMemId = user.memberId.replace(/-M$/, `-${nextChar}`);
      dto.memberId = parentMemId;
    }

    const childCardNumber = `CRD-${Math.floor(1000 + Math.random() * 9000)}`;
    dto.businessAccId = user.createdById;
    dto.childCardNumber = childCardNumber;
    const obj = Object.assign(dto);
    return this.repo.save(obj);
  }

  async findBySearch(dto: SearchChildDto) {
    const keyword = dto.keyword || '';
    const result = await this.repo
      .createQueryBuilder('userChild')
      .select([
        'userChild.id',
        'userChild.memberId',
        'userChild.childCardNumber',
        'userChild.phoneNumber',
      ])
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'userChild.phoneNumber LIKE :keyword OR userChild.memberId LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .getOne();
    if (!result) {
      throw new NotFoundException('Child Not Found!!');
    }
    return result;
  }

  async findOne(id: string) {
    return this.repo.findOne({ where: { id } });
  }

  async getChildQrCode(childId: string) {
    const userProfile = await this.repo
      .createQueryBuilder('userChild')
      .leftJoinAndSelect('userChild.account', 'account')
      .select([
        'userChild.id',
        'userChild.memberId',

        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.createdAt',
      ])
      .where('userChild.id = :id', { id: childId })
      .getOne();
    if (!userProfile) {
      throw new Error('User not found');
    }
    // return userProfile;
    const qrCode = await generateQrCode(userProfile);
    return { qrCode: qrCode };
  }

  async find(dto: ChildPaginationDto, businessAccId: string) {
    const keyword = dto.keyword || '';
    const startDate = new Date(dto.startDate);
    startDate.setHours(0, 0, 0, 0);
    const endDate = new Date(dto.endDate);
    endDate.setHours(23, 59, 59, 999);

    const query = await this.repo
      .createQueryBuilder('userChild')
      .leftJoinAndSelect('userChild.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .select([
        'userChild.id',
        'userChild.businessAccId',
        'userChild.memberId',
        'userChild.oldDepMemberId',
        'userChild.childCardNumber',
        'userChild.name',
        'userChild.email',
        'userChild.phoneNumber',
        'userChild.relation',
        'userChild.martialStatus',
        'userChild.dob',
        'userChild.doc',
        'userChild.profile',
        'userChild.createAt',

        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.createdAt',

        'userDetail.id',
        'userDetail.formNum',
        'userDetail.memberId',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.email',
        'userDetail.gender',
        'userDetail.profile',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.cardNumber',
        'userDetail.salutation',
        'userDetail.haveBusiness',
        'userDetail.status',
        'userDetail.paymentStatus',
        'userDetail.maritalStatus',
        'userDetail.billTo',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.validity',
        'membershipCard.price',
        'membershipCard.currencyType',
        'membershipCard.memberCount',
      ])
      .where('userChild.businessAccId = :businessAccId', {
        businessAccId: businessAccId,
      });
    if (dto.status && dto.status.length > 0) {
      query.andWhere('userDetail.status = :status', {
        status: dto.status,
      });
    }
    if (dto.phoneNumber && dto.phoneNumber.length > 0) {
      query.andWhere('userChild.phoneNumber = :phoneNumber', {
        phoneNumber: dto.phoneNumber,
      });
    }
    if (dto.membershipType && dto.membershipType.length > 0) {
      query.andWhere('membershipCard.name = :name', {
        name: dto.membershipType,
      });
    }
    if (dto.memberId && dto.memberId.length > 0) {
      query.andWhere('userChild.memberId = :memberId', {
        memberId: dto.memberId,
      });
    }
    if (dto.startDate && dto.endDate) {
      query.andWhere(
        'userDetail.membershipValidTo >= :startDate AND userDetail.membershipValidTo <= :endDate',
        {
          startDate: startDate,
          endDate: endDate,
        },
      );
    }
    const [result, total] = await query
      .orderBy({ 'userChild.createAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async findForCsv(dto: CSVChildPaginationDto, businessAccId: string) {
    const keyword = dto.keyword || '';
    const startDate = new Date(dto.startDate);
    startDate.setHours(0, 0, 0, 0);
    const endDate = new Date(dto.endDate);
    endDate.setHours(23, 59, 59, 999);

    const query = await this.repo
      .createQueryBuilder('userChild')
      .leftJoinAndSelect('userChild.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .select([
        'userChild.id',
        'userChild.businessAccId',
        'userChild.memberId',
        'userChild.oldDepMemberId',
        'userChild.childCardNumber',
        'userChild.name',
        'userChild.email',
        'userChild.phoneNumber',
        'userChild.relation',
        'userChild.martialStatus',
        'userChild.dob',
        'userChild.doc',
        'userChild.profile',
        'userChild.createAt',

        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.createdAt',

        'userDetail.id',
        'userDetail.formNum',
        'userDetail.memberId',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.email',
        'userDetail.gender',
        'userDetail.profile',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.cardNumber',
        'userDetail.salutation',
        'userDetail.haveBusiness',
        'userDetail.status',
        'userDetail.paymentStatus',
        'userDetail.maritalStatus',
        'userDetail.billTo',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.validity',
        'membershipCard.price',
        'membershipCard.currencyType',
        'membershipCard.memberCount',
      ])
      .where('userChild.businessAccId = :businessAccId', {
        businessAccId: businessAccId,
      });
    if (dto.status && dto.status.length > 0) {
      query.andWhere('userDetail.status = :status', {
        status: dto.status,
      });
    }
    if (dto.phoneNumber && dto.phoneNumber.length > 0) {
      query.andWhere('userChild.phoneNumber = :phoneNumber', {
        phoneNumber: dto.phoneNumber,
      });
    }
    if (dto.membershipType && dto.membershipType.length > 0) {
      query.andWhere('membershipCard.name = :name', {
        name: dto.membershipType,
      });
    }
    if (dto.memberId && dto.memberId.length > 0) {
      query.andWhere('userChild.memberId = :memberId', {
        memberId: dto.memberId,
      });
    }
    if (dto.startDate && dto.endDate) {
      query.andWhere(
        'userDetail.membershipValidTo >= :startDate AND userDetail.membershipValidTo <= :endDate',
        {
          startDate: startDate,
          endDate: endDate,
        },
      );
    }
    const [result, total] = await query
      .orderBy({ 'userChild.createAt': 'DESC' })
      .getManyAndCount();

    return { result, total };
  }

  async downloadChildCSV(dto: CSVChildPaginationDto, accountId: string) {
    const formatDate = (dateString) => {
      const date = new Date(dateString);
      const year = date.getFullYear();
      const month = String(date.getMonth() + 1).padStart(2, '0');
      const day = String(date.getDate()).padStart(2, '0');
      return `${day}-${month}-${year}`;
    };

    const { result } = await this.findForCsv(dto, accountId);

    const csvStringifier = createObjectCsvStringifier({
      header: [
        // { id: 'createdAt', title: 'Created At' },
        { id: 'memberId', title: 'Dependent Member Id' },
        { id: 'phoneNumber', title: 'Phone Number' },
        { id: 'email', title: 'Email' },
        { id: 'name', title: 'Name' },
        { id: 'childCardNumber', title: 'Card Number' },
        { id: 'dob', title: 'Date of Birth' },
        { id: 'martialStatus', title: 'Marital Status' },
        { id: 'parentName', title: 'Parent Name' },
        { id: 'relation', title: 'Relation' },
        { id: 'profile', title: 'Profile URL' },
        { id: 'doc', title: 'Document URL' },
        { id: 'membershipValidFrom', title: 'Membership From' },
        { id: 'membershipValidTo', title: 'Membership To' },
        { id: 'status', title: 'Status' },
        { id: 'paymentStatus', title: 'Payment Status' },

        // Separate fields for Membership Card
        { id: 'membershipCardName', title: 'Membership' },
        { id: 'price', title: 'Card Price' },
        { id: 'dependentCount', title: 'Dependent Count' },
      ],
    });

    const records = result.map((child) => {
      const userDetail = child.account['userDetail'];

      const baseRecord = {
        // createdAt: formatDate(account.createdAt),
        memberId: child.memberId,
        phoneNumber: child.phoneNumber,
        email: child.email,
        name: child.name,
        childCardNumber: child.childCardNumber,
        dob: child.dob,
        martialStatus: child.martialStatus,
        parentName: `${userDetail[0].fName} ${userDetail[0].mName ? `${userDetail[0].mName} ${userDetail[0].lName}` : userDetail[0].lName}`,
        relation: child.relation,
        profile: child.profile,
        doc: child.doc,
        membershipValidFrom: userDetail[0].membershipValidFrom,
        membershipValidTo: userDetail[0].membershipValidTo,
        status: userDetail[0].status,
        paymentStatus: userDetail[0].paymentStatus,

        //membership card details
        membershipCardName: userDetail[0]['membershipCard'].name,
        price: userDetail[0]['membershipCard'].price,
        dependentCount: userDetail[0]['membershipCard'].memberCount,
      };
      return baseRecord;
    });

    const csvContent =
      csvStringifier.getHeaderString() +
      csvStringifier.stringifyRecords(records);

    return csvContent;
  }

  async findChild(id: string) {
    const result = await this.repo
      .createQueryBuilder('userChild')
      .leftJoinAndSelect('userChild.account', 'account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .leftJoinAndSelect('membershipCard.cardAmenities', 'cardAmenities')
      .select([
        'userChild.id',
        'userChild.businessAccId',
        'userChild.memberId',
        'userChild.oldDepMemberId',
        'userChild.childCardNumber',
        'userChild.name',
        'userChild.email',
        'userChild.phoneNumber',
        'userChild.relation',
        'userChild.martialStatus',
        'userChild.dob',
        'userChild.doc',
        'userChild.profile',
        'userChild.createAt',

        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.createdAt',

        'userDetail.id',
        'userDetail.formNum',
        'userDetail.memberId',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.email',
        'userDetail.gender',
        'userDetail.profile',
        'userDetail.address1',
        'userDetail.address2',
        'userDetail.city',
        'userDetail.state',
        'userDetail.zipcode',
        'userDetail.memberDoc',
        'userDetail.businessType',
        'userDetail.businessName',
        'userDetail.businessTurnover',
        'userDetail.businessEmail',
        'userDetail.gstNumber',
        'userDetail.businessDoc',
        'userDetail.businessCity',
        'userDetail.businessState',
        'userDetail.businessZipcode',
        'userDetail.businessPhone',
        'userDetail.cardNumber',
        'userDetail.landMark',
        'userDetail.fatherName',
        'userDetail.dob',
        'userDetail.qualification',
        'userDetail.profession',
        'userDetail.panNumber',
        'userDetail.income',
        'userDetail.salutation',
        'userDetail.haveBusiness',
        'userDetail.businessAddress1',
        'userDetail.businessAddress2',
        'userDetail.pan',
        'userDetail.aadhar',
        'userDetail.aadharNumber',
        'userDetail.status',
        'userDetail.paymentStatus',
        'userDetail.maritalStatus',
        'userDetail.billTo',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.validity',
        'membershipCard.price',
        'membershipCard.currencyType',
        'membershipCard.memberCount',
        'membershipCard.cardDesign',

        'cardAmenities.id',
        'cardAmenities.name',
        'cardAmenities.icon',
        'cardAmenities.desc',
        'cardAmenities.shortDesc',
      ])
      .where('userChild.id = :id', { id: id })
      .getOne();
    if (!result) {
      throw new NotFoundException('Child not found with this id!');
    }
    return result;
  }

  async update(id: string, dto: UpdateUserChildDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Child Not Found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async profileImage(image: string, result: UserChild) {
    if (result.profilePath) {
      const oldPath = join(__dirname, '..', '..', result.profilePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      profile: process.env.PV_CDN_LINK + image,
      profilePath: image,
    });
    return this.repo.save(obj);
  }

  async profileImageByUser(image: string, result: UserChild) {
    if (result.profilePath) {
      const oldPath = join(__dirname, '..', '..', result.profilePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      profile: process.env.PV_CDN_LINK + image,
      profilePath: image,
    });
    return this.repo.save(obj);
  }

  async docByUser(image: string, result: UserChild) {
    if (result.docPath) {
      const oldPath = join(__dirname, '..', '..', result.docPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      doc: process.env.PV_CDN_LINK + image,
      docPath: image,
    });
    return this.repo.save(obj);
  }
}
