import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Query,
  MaxFileSizeValidator,
  ParseFilePipe,
  Put,
  UploadedFile,
  UseInterceptors,
  Res,
} from '@nestjs/common';
import { UserChildService } from './user-child.service';
import {
  ChildPaginationDto,
  CreateByUserChildDto,
  CreateUserChildDto,
  CSVChildPaginationDto,
  SearchChildDto,
} from './dto/create-user-child.dto';
import { UpdateUserChildDto } from './dto/update-user-child.dto';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { FileInterceptor } from '@nestjs/platform-express';
import { diskStorage } from 'multer';
import { extname } from 'path';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Response } from 'express';

@Controller('user-child')
export class UserChildController {
  constructor(private readonly userChildService: UserChildService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.CREATE, 'user_child'])
  create(@Body() dto: CreateUserChildDto, @CurrentUser() user: Account) {
    dto.businessAccId = user.id;
    return this.userChildService.create(dto);
  }

  @Post('add-child')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  addChild(@Body() dto: CreateByUserChildDto, @CurrentUser() user: Account) {
    dto.accountId = user.id;
    return this.userChildService.addChild(dto);
  }

  @Get('search/child')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  // @CheckPermissions([PermissionAction.READ, 'user_child'])
  async findBySearch(@Query() dto: SearchChildDto) {
    return this.userChildService.findBySearch(dto);
  }

  @Get('particular-Child/:childId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  particularChild(@Param('childId') childId: string) {
    return this.userChildService.findChild(childId);
  }

  @Get('qr/:childId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  async getChildQrCode(@Param('childId') childId: string) {
    return this.userChildService.getChildQrCode(childId);
  }

  @Get('list')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'user_child'])
  find(@Query() dto: ChildPaginationDto, @CurrentUser() user: Account) {
    return this.userChildService.find(dto, user.id);
  }

  @Get('business/download-child-csv/:businessAccId')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'user_child'])
  async downloadChildCSV(
    @Query() dto: CSVChildPaginationDto,
    @Param('businessAccId') businessAccId: string,
    @Res() res: Response,
  ) {
    const csvFile = await this.userChildService.downloadChildCSV(
      dto,
      businessAccId,
    );

    res.header('Content-Type', 'text/csv');
    res.attachment('child-list.csv');
    res.send(csvFile);
  }

  @Get('single/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'user_child'])
  findChild(@Param('id') id: string) {
    return this.userChildService.findChild(id);
  }

  @Patch('update/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'user_child'])
  update(@Param('id') id: string, @Body() dto: UpdateUserChildDto) {
    return this.userChildService.update(id, dto);
  }

  @Put('profileImage/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'user_child'])
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads/UserChild/profile',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          return callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async profileImage(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 })],
      }),
    )
    file: Express.Multer.File,
  ) {
    const fileData = await this.userChildService.findOne(id);
    return this.userChildService.profileImage(file.path, fileData);
  }

  @Put('profileImageByUser/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads/UserChild/profile',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          return callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async profileImageByUser(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        // validators: [new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 })],
      }),
    )
    file: Express.Multer.File,
  ) {
    const fileData = await this.userChildService.findOne(id);
    return this.userChildService.profileImageByUser(file.path, fileData);
  }

  @Put('docByUser/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.USER)
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads/UserChild/profile',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          return callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async docByUser(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        // validators: [new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 })],
      }),
    )
    file: Express.Multer.File,
  ) {
    const fileData = await this.userChildService.findOne(id);
    return this.userChildService.docByUser(file.path, fileData);
  }

  @Put('childDoc-admin/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'user_child'])
  @UseInterceptors(
    FileInterceptor('file', {
      storage: diskStorage({
        destination: './uploads/UserChild/profile',
        filename: (req, file, callback) => {
          const randomName = Array(32)
            .fill(null)
            .map(() => Math.round(Math.random() * 16).toString(16))
            .join('');
          return callback(null, `${randomName}${extname(file.originalname)}`);
        },
      }),
    }),
  )
  async docByAdmin(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        // validators: [new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 })],
      }),
    )
    file: Express.Multer.File,
  ) {
    const fileData = await this.userChildService.findOne(id);
    return this.userChildService.docByUser(file.path, fileData);
  }
}
