import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import {
  ChildLogPaginationDto,
  CreateChildLogDto,
} from './dto/create-child-log.dto';
import { UpdateChildLogDto } from './dto/update-child-log.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { ChildLog } from './entities/child-log.entity';
import { Brackets, Repository } from 'typeorm';
import { UserChild } from 'src/user-child/entities/user-child.entity';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { MemberEntranceType, MemberStatus } from 'src/enum';
import { createObjectCsvStringifier } from 'csv-writer';

@Injectable()
export class ChildLogService {
  constructor(
    @InjectRepository(ChildLog) private readonly repo: Repository<ChildLog>,
    @InjectRepository(UserChild)
    private readonly childRepo: Repository<UserChild>,
    @InjectRepository(UserDetail)
    private readonly udRepo: Repository<UserDetail>,
  ) {}

  async create(dto: CreateChildLogDto, accountId: string) {
    const child = await this.childRepo.findOne({
      where: { id: dto.userChildId },
    });
    if (!child) {
      throw new NotFoundException('Child Not Found!!');
    }
    const member = await this.udRepo.findOne({
      where: { accountId: child.accountId },
    });
    if (!member) {
      throw new NotFoundException('Member Not Found!!');
    }
    if (member.status != MemberStatus.ACTIVE) {
      throw new ConflictException('Member is not active!');
    }

    const date = new Date().toLocaleDateString('en-CA');
    const lastCheckIn = await this.repo.findOne({
      where: {
        userChildId: dto.userChildId,
        accountId: accountId,
        date: date,
      },
      order: { entryTime: 'DESC' },
    });
    if (lastCheckIn && lastCheckIn.exitTime == null) {
      throw new ConflictException('Child must exit before checking in again!');
    }

    const currTime = new Date().toLocaleTimeString('en-IN', {
      timeZone: 'Asia/Kolkata',
      hour12: false,
    });
    const obj = Object.assign({
      accountId: accountId,
      userChildId: dto.userChildId,
      name: child.name,
      date: date,
      entryTime: currTime,
      status: MemberEntranceType.ENTERED,
    });
    return this.repo.save(obj);
  }

  async findAll(dto: ChildLogPaginationDto) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('childLog')
      .where('childLog.userChildId = :userChildId', {
        userChildId: dto.userChildId,
      });
    if (dto.status.length > 0) {
      query.andWhere('childLog.status = :status', {
        status: dto.status,
      });
    }
    const [result, total] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'childLog.name LIKE :keyword OR childLog.date LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({
        'childLog.date': 'DESC',
        'childLog.entryTime': 'DESC',
      })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();
    return { result, total };
  }

  async downloadChildLogCSV(dto: ChildLogPaginationDto) {
      const formatDate = (dateString) => {
        const date = new Date(dateString);
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${day}-${month}-${year}`;
      };
      const convertSeconds = (totalSeconds: number) => {
        const hours = Math.floor(totalSeconds / 3600);
        const minutes = Math.floor((totalSeconds % 3600) / 60);
        const seconds = totalSeconds % 60;
        const pad = (value: number): string => value.toString().padStart(2, '0');
        return `${pad(hours)}:${pad(minutes)}:${pad(seconds)}`;
      }
  
      const { result } = await this.findAll(dto);
  
      const csvStringifier = createObjectCsvStringifier({
        header: [
          { id: 'name', title: 'Name' },
          { id: 'date', title: 'Date' },
          { id: 'entryTime', title: 'Entry Time' },
          { id: 'exitTime', title: 'Exit Time' },
          { id: 'duration', title: 'Duration' },
          { id: 'status', title: 'Status' },
        ],
      });
  
      const records = result.map((log) => {
        const baseRecord = {
          name: log.name,
          date: log.date,
          entryTime: log.entryTime,
          exitTime: log.exitTime,
          duration: convertSeconds(log.duration),
          status: log.status,
        };
        return baseRecord;
      });
  
      const csvContent =
        csvStringifier.getHeaderString() +
        csvStringifier.stringifyRecords(records);
  
      return csvContent;
    }

  async update(accountId: string, dto: UpdateChildLogDto) {
    const lastCheckIn = await this.repo.findOne({
      where: {
        userChildId: dto.userChildId,
        accountId: accountId,
        status: MemberEntranceType.ENTERED,
      },
      order: { entryTime: 'DESC' },
    });
    if (!lastCheckIn) {
      throw new NotFoundException('Member not entered yet!');
    }

    const currTime = new Date().toLocaleTimeString('en-IN', {
      timeZone: 'Asia/Kolkata',
      hour12: false,
    });

    const entryTime = lastCheckIn.entryTime;
    const parseTimeString = (timeStr: string): Date => {
      const [hours, minutes, seconds] = timeStr.split(':').map(Number);
      const now = new Date();
      now.setHours(hours, minutes, seconds, 0);
      return now;
    };

    const entryDate = parseTimeString(entryTime);
    const currDate = parseTimeString(currTime);

    // Calculate duration in milliseconds
    const durationMs = currDate.getTime() - entryDate.getTime();
    const durationInSeconds = Math.floor(durationMs / 1000);

    const obj = Object.assign(lastCheckIn, {
      exitTime: currTime,
      duration: durationInSeconds,
      status: MemberEntranceType.EXITED,
    });
    return this.repo.save(obj);
  }
}
