import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Query,
  Res,
} from '@nestjs/common';
import { ChildLogService } from './child-log.service';
import {
  ChildLogPaginationDto,
  CreateChildLogDto,
} from './dto/create-child-log.dto';
import { UpdateChildLogDto } from './dto/update-child-log.dto';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { Response } from 'express';

@Controller('child-log')
export class ChildLogController {
  constructor(private readonly childLogService: ChildLogService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.CREATE, 'child_log'])
  create(@Body() dto: CreateChildLogDto, @CurrentUser() user: Account) {
    return this.childLogService.create(dto, user.id);
  }

  @Get('list')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'child_log'])
  findAll(@Query() dto: ChildLogPaginationDto) {
    return this.childLogService.findAll(dto);
  }

  @Get('download-childlog-csv')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'user_log'])
  async downloadChildLogCSV(
    @Query() dto: ChildLogPaginationDto,
    @Res() res: Response,
  ) {
    const csvFile = await this.childLogService.downloadChildLogCSV(dto);

    res.header('Content-Type', 'text/csv');
    res.attachment('child-log-list.csv');
    res.send(csvFile);
  }

  @Patch('exit-child')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'child_log'])
  update(@CurrentUser() user: Account, @Body() dto: UpdateChildLogDto) {
    return this.childLogService.update(user.id, dto);
  }
}
