import {
  BadRequestException,
  ConflictException,
  Inject,
  Injectable,
  NotFoundException,
  UnauthorizedException,
} from '@nestjs/common';
import {
  BusinessPaginationDto,
  BusinessStatusDto,
  CreateBusinessDto,
  EmailVerifyDto,
  PhoneVerifyDto,
  VerifyBusinessDto,
} from './dto/create-business.dto';
import { UpdateBusinessDto } from './dto/update-business.dto';
import { InjectRepository } from '@nestjs/typeorm';
import { Business } from './entities/business.entity';
import { Brackets, Repository } from 'typeorm';
import { CACHE_MANAGER } from '@nestjs/cache-manager';
import { Cache } from 'cache-manager';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
import { sendOtp } from 'src/utils/sms.utils';
import { BusinessStatus, DefaultStatus, UserRole } from 'src/enum';
import { createObjectCsvStringifier } from 'csv-writer';
import { Setting } from 'src/settings/entities/setting.entity';
import { Licence } from 'src/licence/entities/licence.entity';
import { Menu } from 'src/menus/entities/menu.entity';
import { UserPermission } from 'src/user-permissions/entities/user-permission.entity';
import { MenusService } from 'src/menus/menus.service';
import { PermissionsService } from 'src/permissions/permissions.service';
import { UserPermissionsService } from 'src/user-permissions/user-permissions.service';
import { AdminDetail } from 'src/admin-detail/entities/admin-detail.entity';
import { unlink } from 'fs/promises';
import { join } from 'path';
import { Account } from 'src/account/entities/account.entity';

@Injectable()
export class BusinessService {
  constructor(
    @InjectRepository(Business) private readonly repo: Repository<Business>,
    @InjectRepository(Account) private readonly accRepo: Repository<Account>,
    @InjectRepository(AdminDetail)
    private readonly adminDetailepo: Repository<AdminDetail>,
    @InjectRepository(Licence)
    private readonly licenceRepo: Repository<Licence>,
    @InjectRepository(Setting)
    private readonly settingRepo: Repository<Setting>,
    @InjectRepository(UserPermission)
    private readonly userPermRepo: Repository<UserPermission>,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
    private readonly nodeMailerService: NodeMailerService,
    private readonly menuService: MenusService,
    private readonly permissionService: PermissionsService,
    private readonly userPermService: UserPermissionsService,
  ) {}

  async getSmsOfMasterAdmin(businessAccId: string) {
    const smsCred = await this.adminDetailepo.findOne({
      where: { accountId: businessAccId },
    });
    return {
      smsRoute: smsCred.smsRoute,
      smsSenderId: smsCred.smsSenderId,
      smsApiKey: smsCred.smsApiKey,
    };
  }

  async getSmsOfBusinessAdmin(businessAccId: string) {
    const smsCred = await this.repo.findOne({
      where: { accountId: businessAccId },
    });
    return {
      smsRoute: smsCred.smsRoute,
      smsSenderId: smsCred.smsSenderId,
      smsApiKey: smsCred.smsApiKey,
    };
  }

  async sendOTPEmail(dto: EmailVerifyDto, accountId: string) {
    const result = await this.accRepo.findOne({
      where: { email: dto.email, roles: UserRole.BUSINESS },
    });
    if (result) {
      throw new ConflictException('Email Already exists!');
    }
    // const otp = 783200;
    const otp = Math.floor(100000 + Math.random() * 900000);
    this.nodeMailerService.sendOtpInEmailForMaster(dto.email, otp, accountId);
    this.cacheManager.set(dto.email, otp, 10 * 60 * 1000); // for 10 mins
    return { message: 'OTP sent to the email!' };
  }

  async verifyEmail(dto: EmailVerifyDto) {
    const storedOtp = await this.cacheManager.get<string>(dto.email);
    if (!storedOtp || storedOtp !== dto.otp) {
      throw new BadRequestException('Invalid or expired OTP');
    }
    return { message: 'OTP Matched!' };
  }

  async sendOTPPhone(dto: PhoneVerifyDto, accountId: string) {
    const result = await this.accRepo.findOne({
      where: { phoneNumber: dto.phoneNumber, roles: UserRole.BUSINESS },
    });
    if (result) {
      throw new ConflictException('Phone Number Already exists!');
    }
    // const otp = 783200;
    const smsConfig = await this.getSmsOfMasterAdmin(accountId);
    const otp = Math.floor(100000 + Math.random() * 900000);
    sendOtp(dto.phoneNumber, otp, smsConfig);
    this.cacheManager.set(dto.phoneNumber, otp, 10 * 60 * 1000);
    return { message: 'OTP sent to the Phone Number!' };
  }

  async sendOTPPhoneByBusiness(dto: PhoneVerifyDto, accountId: string) {
    const check = await this.accRepo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .where(
        'account.phoneNumber = :phoneNumber AND account.roles = :roles AND userDetail.createdById = :createdById',
        {
          phoneNumber: dto.phoneNumber,
          roles: UserRole.USER,
          createdById: accountId,
        },
      )
      .getOne();
    if (check) {
      throw new BadRequestException('User already exists with this number!');
    }

    // const otp = 783200;
    const smsConfig = await this.getSmsOfBusinessAdmin(accountId);
    const otp = Math.floor(100000 + Math.random() * 900000);
    sendOtp(dto.phoneNumber, otp, smsConfig);
    this.cacheManager.set(dto.phoneNumber, otp, 10 * 60 * 1000);
    return { message: 'OTP sent to the Phone Number!' };
  }

  async verifyPhone(dto: PhoneVerifyDto) {
    const storedOtp = await this.cacheManager.get<string>(dto.phoneNumber);
    if (!storedOtp || storedOtp !== dto.otp) {
      throw new BadRequestException('Invalid or expired OTP');
    }
    return { message: 'OTP Matched!' };
  }

  async create(dto: CreateBusinessDto) {
    const result = await this.repo.findOne({
      where: { accountId: dto.accountId },
    });
    if (result) {
      const obj = Object.assign(result, dto);
      const business = await this.repo.save(obj);
      const settingObj = Object.assign({
        accountId: dto.accountId,
        title: 'Setting',
        user_domain: null,
        admin_domain: null,
        mobile_domain: null,
        dateFormat: 'd-m-y',
        timeFormat: '12hours',
        timeZone: 'Asia/Kolkata',
        currency: 'INR',
      });
      await this.settingRepo.save(settingObj);
      return business;
    } else {
      const obj = Object.assign(dto);
      return this.repo.save(obj);
    }
  }

  async findAll(dto: BusinessPaginationDto) {
    const keyword = dto.keyword || '';

    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 999);

    const query = await this.repo
      .createQueryBuilder('business')
      .leftJoinAndSelect('business.account', 'account')
      .leftJoinAndSelect('business.licence', 'licence')
      .select([
        'business.id',
        'business.gender',
        'business.personName',
        'business.personDesignation',
        'business.personEmail',
        'business.personPhone',
        'business.businessKey',
        'business.businessType',
        'business.businessName',
        'business.parentCompanyName',
        'business.businessPhone',
        'business.businessEmail',
        'business.gstNo',
        'business.address1',
        'business.address2',
        'business.zipCode',
        'business.city',
        'business.state',
        'business.country',
        'business.signatory',
        'business.logo',
        'business.bwLogo',
        'business.brandLogo',
        'business.doc1',
        'business.doc2',
        'business.gstCertificate',
        'business.workOrder',
        'business.status',
        'business.accountId',
        'business.createdAt',
        'business.updatedAt',

        'account.id',
        'account.phoneNumber',
        'account.email',
        'account.password',
        'account.roles',
        'account.createdAt',

        'licence.id',
        'licence.businessId',
        'licence.userLimit',
        'licence.licenceKey',
        'licence.activationKey',
        'licence.startDate',
        'licence.renewalDate',
        'licence.amc',
        'licence.createdAt',
        'licence.status',
      ])
      .where('business.status = :status', { status: dto.status });
    if (dto.fromDate && dto.toDate) {
      query.andWhere(
        'business.createdAt >= :fromDate AND business.createdAt <= :toDate',
        {
          fromDate: fromDate,
          toDate: toDate,
        },
      );
    }

    const [result, total] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.phoneNumber LIKE :keyword OR business.gstNo LIKE :keyword OR business.businessName LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'business.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();
    return { result, total };
  }

  async findOne(id: string) {
    const result = await this.repo
      .createQueryBuilder('business')
      .leftJoinAndSelect('business.account', 'account')
      .leftJoinAndSelect('account.businessContract', 'businessContract')
      .leftJoinAndSelect('businessContract.contract', 'contract')
      .leftJoinAndSelect('business.licence', 'licence')
      .select([
        'business.id',
        'business.gender',
        'business.personName',
        'business.personDesignation',
        'business.personEmail',
        'business.personPhone',
        'business.businessKey',
        'business.businessType',
        'business.businessName',
        'business.parentCompanyName',
        'business.businessPhone',
        'business.businessEmail',
        'business.gstNo',
        'business.address1',
        'business.address2',
        'business.zipCode',
        'business.city',
        'business.state',
        'business.country',
        'business.signatory',
        'business.logo',
        'business.brandLogo',
        'business.doc1',
        'business.doc2',
        'business.gstCertificate',
        'business.workOrder',
        'business.status',
        'business.accountId',
        'business.createdAt',
        'business.updatedAt',

        'account.id',
        'account.phoneNumber',
        'account.email',
        'account.password',
        'account.roles',
        'account.createdAt',

        'businessContract.id',
        'contract.id',
        'contract.contractName',
        'contract.contractTypeId',
        'contract.validFrom',
        'contract.validTill',
        'contract.desc',
        'contract.status',

        'licence.id',
        'licence.businessId',
        'licence.userLimit',
        'licence.licenceKey',
        'licence.activationKey',
        'licence.startDate',
        'licence.renewalDate',
        'licence.amc',
        'licence.createdAt',
        'licence.status',
      ])
      .where('business.id = :id', { id: id })
      .getOne();
    return result;
  }

  async findBusiness(accountId: string) {
    return this.repo.findOne({ where: { accountId } });
  }

  async downloadBusinessCSV(dto: BusinessPaginationDto): Promise<string> {
    const { result } = await this.findAll(dto);

    const flattenedData = result.flatMap((business) => {
      if (!business.licence || business.licence.length === 0) {
        return [
          {
            ...business,
            licenceId: '',
            licenceKey: '',
            activationKey: '',
            userLimit: 0,
            startDate: null,
            renewalDate: null,
            amc: '',
            licenceStatus: '',
          },
        ];
      }

      return business.licence.map((licence) => ({
        ...business,
        licenceId: licence.id,
        licenceKey: licence.licenceKey,
        activationKey: licence.activationKey,
        userLimit: licence.userLimit,
        startDate: licence.startDate,
        renewalDate: licence.renewalDate,
        amc: licence.amc,
        licenceStatus: licence.status,
      }));
    });

    const csvStringifier = createObjectCsvStringifier({
      header: [
        // { id: 'id', title: 'ID' },
        { id: 'personName', title: 'Person Name' },
        { id: 'gender', title: 'Gender' },
        { id: 'personEmail', title: 'Email' },
        { id: 'personPhone', title: 'Phone' },
        { id: 'businessKey', title: 'Business Key' },
        { id: 'businessType', title: 'Business Type' },
        { id: 'businessName', title: 'Business Name' },
        { id: 'gstNo', title: 'GST Number' },
        { id: 'address1', title: 'Address 1' },
        { id: 'address2', title: 'Address 2' },
        { id: 'zipCode', title: 'Zip Code' },
        { id: 'city', title: 'City' },
        { id: 'state', title: 'State' },
        { id: 'country', title: 'Country' },
        { id: 'signatory', title: 'Signatory' },
        { id: 'logo', title: 'Logo' },
        { id: 'brandLogo', title: 'Brand Logo' },
        { id: 'doc1', title: 'Doc1' },
        { id: 'doc2', title: 'Doc2' },
        { id: 'gstCertificate', title: 'GST Certificate' },
        { id: 'workOrder', title: 'Work Order' },
        { id: 'status', title: 'Status' },
        { id: 'createdAt', title: 'Created At' },

        // { id: 'licenceId', title: 'Licence ID' },
        { id: 'licenceKey', title: 'Licence Key' },
        { id: 'activationKey', title: 'Activation Key' },
        { id: 'userLimit', title: 'User Limit' },
        { id: 'startDate', title: 'Licence Start Date' },
        { id: 'renewalDate', title: 'Licence Renewal Date' },
        { id: 'amc', title: 'AMC' },
        { id: 'licenceStatus', title: 'Licence Status' },
      ],
    });

    // Convert data to CSV format
    const csvContent =
      csvStringifier.getHeaderString() +
      csvStringifier.stringifyRecords(flattenedData);
    return csvContent;
  }

  async pdf(dto: BusinessPaginationDto) {
    const keyword = dto.keyword || '';

    const fromDate = new Date(dto.fromDate);
    fromDate.setHours(0, 0, 0, 0);

    const toDate = new Date(dto.toDate);
    toDate.setHours(23, 59, 59, 999);

    const query = await this.repo
      .createQueryBuilder('business')
      .leftJoinAndSelect('business.licence', 'licence')
      .select([
        'business.id',
        'business.gender',
        'business.personName',
        'business.personDesignation',
        'business.personEmail',
        'business.personPhone',
        'business.businessKey',
        'business.businessType',
        'business.businessName',
        'business.gstNo',
        'business.address1',
        'business.address2',
        'business.zipCode',
        'business.city',
        'business.state',
        'business.country',
        'business.signatory',
        'business.logo',
        'business.brandLogo',
        'business.doc1',
        'business.doc2',
        'business.gstCertificate',
        'business.workOrder',
        'business.status',
        'business.accountId',
        'business.createdAt',
        'business.updatedAt',

        'licence.id',
        'licence.businessId',
        'licence.userLimit',
        'licence.licenceKey',
        'licence.activationKey',
        'licence.startDate',
        'licence.renewalDate',
        'licence.amc',
        'licence.createdAt',
        'licence.status',
      ])
      .where('business.status = :status', { status: dto.status });
    if (dto.fromDate && dto.toDate) {
      query.andWhere(
        'business.createdAt >= :fromDate AND business.createdAt <= :toDate',
        {
          fromDate: fromDate,
          toDate: toDate,
        },
      );
    }

    const result = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'business.personPhone LIKE :keyword OR business.gstNo LIKE :keyword OR business.businessName LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'business.createdAt': 'DESC' })
      .take(dto.limit)
      .offset(dto.offset)
      .getMany();
    return result;
  }

  async acitveBusiness(dto: VerifyBusinessDto, accountId: string) {
    const business = await this.repo.findOne({ where: { accountId } });
    if (!business) {
      throw new NotFoundException('Business not found!');
    }
    if (business.businessName != dto.businessName) {
      throw new NotFoundException('Please enter correct business name!');
    }
    const licence = await this.licenceRepo.findOne({
      where: { businessId: business.id },
    });
    if (!licence) {
      throw new NotFoundException('License not found for this business!');
    }
    if (
      dto.activationKey == licence.activationKey &&
      dto.licenceKey == licence.licenceKey
    ) {
      if (business.status == BusinessStatus.ACTIVE) {
        return { message: 'You are already active' };
      }
      const obj = Object.assign(business, { status: BusinessStatus.ACTIVE });
      await this.repo.save(obj);

      const acc = await this.accRepo.findOne({ where: { id: accountId } });
      await this.nodeMailerService.sendActivationMailFromMaster(
        acc.email,
        `ab3ee683-cda8-11ef-b19b-54e1ad3f3f66`,
      );

      return { message: 'Activation Successful!' };
    } else {
      throw new NotFoundException('Invalid activation or license key!');
    }
  }

  async update(id: string, dto: UpdateBusinessDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Business Not Found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async updateByBusiness(accountId: string, dto: UpdateBusinessDto) {
    const result = await this.repo.findOne({ where: { accountId } });
    if (!result) {
      throw new NotFoundException('Business Not Found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async genBusinessKey(accountId: string) {
    const result = await this.repo.findOne({ where: { accountId } });
    if (!result) {
      throw new NotFoundException('Business not found!');
    }
    const date = new Date().getDate();
    const month = new Date().getMonth() + 1;
    const fourDigitNumb = Math.floor(1000 + Math.random() * 9000);
    const businessKey = `PRI${date}${month}${fourDigitNumb}`;

    const obj = Object.assign(result, { businessKey: businessKey });
    return this.repo.save(obj);
  }

  async document1(image: string, result: Business) {
    if (result.doc1Path) {
      const oldPath = join(__dirname, '..', '..', result.doc1Path);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      doc1: process.env.PV_CDN_LINK + image,
      doc1Path: image,
    });
    return this.repo.save(obj);
  }

  async document2(image: string, result: Business) {
    if (result.doc2Path) {
      const oldPath = join(__dirname, '..', '..', result.doc2Path);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      doc2: process.env.PV_CDN_LINK + image,
      doc2Path: image,
    });
    return this.repo.save(obj);
  }

  async gstCertificate(image: string, result: Business) {
    if (result.gstCertificatePath) {
      const oldPath = join(__dirname, '..', '..', result.gstCertificatePath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      gstCertificate: process.env.PV_CDN_LINK + image,
      gstCertificatePath: image,
    });
    return this.repo.save(obj);
  }

  async workOrder(image: string, result: Business) {
    if (result.workOrderPath) {
      const oldPath = join(__dirname, '..', '..', result.workOrderPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      workOrder: process.env.PV_CDN_LINK + image,
      workOrderPath: image,
    });
    return this.repo.save(obj);
  }

  async logo(image: string, result: Business) {
    if (result.logoPath) {
      const oldPath = join(__dirname, '..', '..', result.logoPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      logo: process.env.PV_CDN_LINK + image,
      logoPath: image,
    });
    return this.repo.save(obj);
  }

  async bwLogo(image: string, result: Business) {
    if (result.bwLogoPath) {
      const oldPath = join(__dirname, '..', '..', result.bwLogoPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      bwLogo: process.env.PV_CDN_LINK + image,
      bwLogoPath: image,
    });
    return this.repo.save(obj);
  }

  async brandLogo(image: string, result: Business) {
    if (result.brandLogoPath) {
      const oldPath = join(__dirname, '..', '..', result.brandLogoPath);
      try {
        await unlink(oldPath);
      } catch (err) {
        console.warn(`Failed to delete old image: ${oldPath}`, err.message);
      }
    }
    const obj = Object.assign(result, {
      brandLogo: process.env.PV_CDN_LINK + image,
      brandLogoPath: image,
    });
    return this.repo.save(obj);
  }

  async status(id: string, dto: BusinessStatusDto, masterAccId: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Business Not Found!');
    }
    const obj = Object.assign(result, { status: dto.status });
    const status = await this.repo.save(obj);
    const businessAcc = await this.accRepo.findOne({
      where: { id: result.accountId },
    });
    if (dto.status == BusinessStatus.ACTIVE) {
      await this.nodeMailerService.sendActivationMailFromMaster(
        businessAcc.email,
        masterAccId,
      );
    } else {
      await this.nodeMailerService.sendDeactivationMailFromMaster(
        businessAcc.email,
        masterAccId,
      );
    }
    return status;
  }
}
