import {
  Controller,
  Get,
  Post,
  Body,
  Patch,
  Param,
  Delete,
  UseGuards,
  Query,
  Res,
} from '@nestjs/common';
import { BusinessPaymentRecordService } from './business-payment-record.service';
import {
  CreateBusinessPaymentRecordDto,
  RecordPaginationDto,
} from './dto/create-business-payment-record.dto';
import { UpdateBusinessPaymentRecordDto } from './dto/update-business-payment-record.dto';
import { AuthGuard } from '@nestjs/passport';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { UserRole } from 'src/enum';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { Account } from 'src/account/entities/account.entity';
import { Response } from 'express';
import { createMasterInvoice } from 'src/utils/masterAdminInvoice.utils';

@Controller('business-payment-record')
export class BusinessPaymentRecordController {
  constructor(
    private readonly businessPaymentRecordService: BusinessPaymentRecordService,
  ) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  create(
    @Body() dto: CreateBusinessPaymentRecordDto,
    @CurrentUser() user: Account,
  ) {
    return this.businessPaymentRecordService.create(dto, user.id);
  }

  @Get('list')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  findAll(@Query() dto: RecordPaginationDto) {
    return this.businessPaymentRecordService.findAll(dto);
  }

  @Get('particular/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  findDetail(@Param('id') id: string){
    return this.businessPaymentRecordService.findDetail(id)
  }

  @Get('invoice/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  async findOne(@Param('id') id: string, @Res() res: Response) {
    // 
    const details = await this.businessPaymentRecordService.findOne(id);
    const invoice = {
      transactionId: details.transactionId,
      invoiceNumber: details.invoiceNumber,
      subject: details.subject,
      businessName: details.businessName,
      subTotal: details.subTotal,
      total: details.total,
      paid: details.paid,
      amountDue: details.amountDue,
      paymentMode: details.paymentMode,
      paymentDate: details.paymentDate,
      note: details.note,
      createdAt: details.createdAt,
      business: {
        personName: details.account['business']['0']['personName'],
        gstNo: details.account['business']['0']['gstNo'],
      },
      tax: details.account['tax'],
    };
    const pdf = await createMasterInvoice(invoice);
    const name = Date.now().toString() + `-invoice_${invoice.businessName}.pdf`;
    res.setHeader('Content-Type', 'application/pdf');
    res.set('Content-Disposition', `attachment; filename="${name}"`);
    pdf.pipe(res);
    pdf.end();
    // return details;
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  update(@Param('id') id: string, @Body() dto: UpdateBusinessPaymentRecordDto) {
    return this.businessPaymentRecordService.update(id, dto);
  }

  @Delete('remove/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN)
  remove(@Param('id') id: string) {
    return this.businessPaymentRecordService.remove(id);
  }
}
