import { HttpService } from '@nestjs/axios';
import { CACHE_MANAGER } from '@nestjs/cache-manager';
import {
  BadRequestException,
  ConflictException,
  Inject,
  Injectable,
  NotFoundException,
  UnauthorizedException,
} from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { InjectRepository } from '@nestjs/typeorm';
import * as bcrypt from 'bcrypt';
import { Cache } from 'cache-manager';
import { Account } from 'src/account/entities/account.entity';
import {
  DefaultStatus,
  LogType,
  LoginType,
  MemberStatus,
  UserRole,
} from 'src/enum';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { UserPermission } from 'src/user-permissions/entities/user-permission.entity';
import APIFeatures from 'src/utils/apiFeatures.utils';
import { Repository } from 'typeorm';
import {
  AdminSigninDto,
  BusinessCreateDto,
  BusinessForgotDto,
  BusinessLoginDto,
  BusinessResetDto,
  EmailDto,
  ForgotPassDto,
  MemberSigninDto,
  OtpDto,
  SigninDto,
  StaffLoginDto,
  VerifyOtpDto,
} from './dto/login.dto';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
import { LoginHistory } from 'src/login-history/entities/login-history.entity';
import { Business } from 'src/business/entities/business.entity';
import { UserChild } from 'src/user-child/entities/user-child.entity';
import { sendOtp } from 'src/utils/sms.utils';

@Injectable()
export class AuthService {
  constructor(
    private jwtService: JwtService,
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(UserPermission)
    private readonly upRepo: Repository<UserPermission>,
    @InjectRepository(Business)
    private readonly businessRepo: Repository<Business>,
    @InjectRepository(UserDetail)
    private readonly userDetailRepo: Repository<UserDetail>,
    @InjectRepository(UserChild)
    private readonly userChildRepo: Repository<UserChild>,
    @InjectRepository(LoginHistory)
    private readonly logRepo: Repository<LoginHistory>,
    private readonly httpService: HttpService,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
    private readonly nodeMailerService: NodeMailerService,
  ) {}

  async adminLogin(dto: AdminSigninDto) {
    const admin = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.adminDetail', 'adminDetail')
      .where('account.email = :email AND account.roles = :roles', {
        email: dto.email,
        roles: UserRole.ADMIN,
      })
      .getOne();
    if (!admin) {
      throw new NotFoundException('EmailId not found! Please contact admin.');
    }
    const comparePassword = await bcrypt.compare(dto.password, admin.password);
    if (!comparePassword) {
      throw new UnauthorizedException('password mismatched!!');
    }
    const otp = 783200;
    // const otp = Math.floor(100000 + Math.random() * 900000);
    // this.nodeMailerService.sendOtpInEmailForMaster(dto.email, otp,`ab3ee683-cda8-11ef-b19b-54e1ad3f3f66`);
    this.cacheManager.set(dto.email, otp, 10 * 60 * 1000);

    return 'OTP Sent in email!';
  }

  async verifyOtp(dto: VerifyOtpDto, ip: string) {
    const admin = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.adminDetail', 'adminDetail')
      .where('account.email = :email AND account.roles = :roles', {
        email: dto.email,
        roles: UserRole.ADMIN,
      })
      .getOne();
    if (!admin) {
      throw new NotFoundException('Admin account not found');
    }
    const storedOtp = await this.cacheManager.get<string>(dto.email);
    if (!storedOtp || storedOtp !== dto.otp) {
      throw new BadRequestException('Invalid or expired OTP');
    }
    const token = await APIFeatures.assignJwtToken(admin.id, this.jwtService);

    const obj = Object.create({
      loginId: admin.email,
      ip: ip,
      type: LogType.LOGIN,
      accountId: admin.id,
    });
    await this.logRepo.save(obj);

    return { token, admin: { id: admin.id, email: admin.email } };
  }

  async businessLogin(dto: BusinessLoginDto) {
    const result = await this.repo
      .createQueryBuilder('account')
      .where('account.email = :email AND account.roles = :roles', {
        email: dto.email,
        roles: UserRole.BUSINESS,
      })
      .getOne();
    if (!result) {
      throw new NotFoundException('EmailId not found! Please contact admin.');
    }
    const comparePassword = await bcrypt.compare(dto.password, result.password);
    if (!comparePassword) {
      throw new UnauthorizedException('password mismatched!!');
    }
    const otp = 783200;
    // const otp = Math.floor(100000 + Math.random() * 900000);
    // const mail = await this.nodeMailerService.sendOtpInMail(dto.email, otp, dto.accountId);
    this.cacheManager.set(dto.email, otp, 10 * 60 * 1000);
    return 'OTP Sent in email!';
  }

  async businessVerifyOTP(dto: VerifyOtpDto) {
    const business = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .where('account.email = :email AND account.roles = :roles', {
        email: dto.email,
        roles: UserRole.BUSINESS,
      })
      .getOne();
    if (!business) {
      throw new NotFoundException('Business account not found');
    }
    const storedOtp = await this.cacheManager.get<string>(dto.email);
    if (!storedOtp || storedOtp !== dto.otp) {
      throw new BadRequestException('Invalid or expired OTP');
    }

    const token = await APIFeatures.assignJwtToken(
      business.id,
      this.jwtService,
    );
    return {
      token,
      roles: UserRole.BUSINESS,
      business: {
        id: business.id,
        email: business.email,
        status: business.business[0].status,
      },
    };
  }

  async staffLogin(dto: StaffLoginDto) {
    const staff = await this.repo.findOne({
      where: { email: dto.email, roles: UserRole.STAFF },
    });
    if (!staff) {
      throw new NotFoundException('Staff not Found ');
    }
    if (staff.status != DefaultStatus.ACTIVE) {
      throw new UnauthorizedException(
        'Staff is not active. Pleasecontact to admin!',
      );
    }
    const comparePassword = await bcrypt.compare(dto.password, staff.password);
    if (!comparePassword) {
      throw new UnauthorizedException('Invalid Password');
    }

    const otp = 783200;
    // const otp = Math.floor(100000 + Math.random() * 900000);
    // this.nodeMailerService.sendOtpInMail(dto.email, otp, dto.accountId);
    this.cacheManager.set(dto.email, otp, 10 * 60 * 1000);

    return 'OTP Sent in email!';
  }

  async staffVerifyOTP(dto: VerifyOtpDto) {
    const staff = await this.repo.findOne({
      where: { email: dto.email, roles: UserRole.STAFF },
    });
    if (!staff) {
      throw new NotFoundException('Staff not found!');
    }
    const storedOtp = await this.cacheManager.get<string>(dto.email);
    if (!storedOtp || storedOtp !== dto.otp) {
      throw new BadRequestException('Invalid or expired OTP');
    }

    const token = await APIFeatures.assignJwtToken(staff.id, this.jwtService);
    return {
      token,
      roles: UserRole.STAFF,
      account: {
        id: staff.id,
        email: staff.email,
        status: staff.status,
      },
    };
  }

  async businessforgotPass(dto: BusinessForgotDto) {
    const result = await this.repo.findOne({
      where: { email: dto.email, roles: UserRole.BUSINESS },
    });
    if (!result) {
      throw new NotFoundException('Account not found!');
    }
    // const otp = 783200;
    const otp = Math.floor(100000 + Math.random() * 900000);
    this.nodeMailerService.sendOtpInMail(dto.email, otp, dto.accountId);
    this.cacheManager.set(dto.email, otp, 10 * 60 * 1000);

    return 'OTP Sent in email!';
  }

  async businessOtpCheck(email: string, otp: string) {
    const storedOtp = await this.cacheManager.get<string>(email);
    if (storedOtp !== otp) {
      throw new BadRequestException('Invalid or expired OTP');
    }
    return { message: 'OTP Macthed.' };
  }

  async businessResetPass(dto: BusinessResetDto) {
    const business = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .where('account.email = :email AND account.roles = :roles', {
        email: dto.email,
        roles: UserRole.BUSINESS,
      })
      .getOne();
    if (!business) {
      throw new NotFoundException('Business account not found');
    }
    const hashedPassword = await bcrypt.hash(dto.newPassword, 10);
    business.password = hashedPassword;

    await this.repo.save(business);
    await this.cacheManager.del(dto.email);

    return { message: 'Password reset successfully' };
  }

  async memberLogin(dto: MemberSigninDto) {
    let user = await this.repo.findOne({
      where: { phoneNumber: dto.loginId, roles: UserRole.USER },
    });
    if (!user) {
      const obj = Object.create({
        phoneNumber: dto.loginId,
        type: LoginType.PHONE,
        roles: UserRole.USER,
        // fcm,
      });
      user = await this.repo.save(obj);
      const findUd = await this.userDetailRepo.findOne({
        where: {},
        select: ['formNum'],
        order: { formNum: 'DESC' },
      });
      const udObj = Object.create({
        accountId: user.id,
        createdById: dto.businessAccId,
        status: dto.status,
        formNum: dto.status == MemberStatus.ENQUIRY ? !findUd || !findUd.formNum ? 203 : findUd.formNum + 1 : null,
      });
      await this.userDetailRepo.save(udObj);
    }
    // const fcmObj = Object.assign(user,{fcm: fcm});
    // await this.repo.save(fcmObj);

    const smsConfig = await this.getSmsOfBusiness(dto.businessAccId);

    // const otp = 783200;
    const otp = Math.floor(100000 + Math.random() * 900000);
    sendOtp(dto.loginId, otp, smsConfig);
    this.cacheManager.set(dto.loginId, otp, 10 * 60 * 1000);
    return { loginId: dto.loginId };
  }

  async memberVerifyOtp(dto: OtpDto) {
    const user = await this.getUserDetails(dto.loginId, UserRole.USER);
    const sentOtp = await this.cacheManager.get(dto.loginId);

    // if (phoneNumber != '8092326469') {
    if (dto.otp != sentOtp) {
      throw new UnauthorizedException('Invalid otp!');
    }
    // }
    const token = await APIFeatures.assignJwtToken(user.id, this.jwtService);

    const check = user.userDetail[0].membershipCardId;
    if (check == null) {
      return {
        token,
        latest: true,
        status: user.userDetail[0].status,
      };
    } else {
      return {
        token,
        latest: false,
        status: user.userDetail[0].status,
      };
    }
  }

  async childLogin(dto: SigninDto) {
    const child = await this.userChildRepo.findOne({
      where: { phoneNumber: dto.loginId },
    });
    if (!child) {
      throw new NotFoundException('Child Not Found!');
    }

    const smsConfig = await this.getSmsOfBusiness(
      `6cc2f800-4899-4fae-b4b0-b0c7894e6c32`,
    );

    // const otp = 783200;
    const otp = Math.floor(100000 + Math.random() * 900000);
    sendOtp(dto.loginId, otp, smsConfig);
    this.cacheManager.set(dto.loginId, otp, 10 * 60 * 1000);
    return { loginId: dto.loginId };
  }

  async childVerifyOtp(dto: OtpDto) {
    const child = await this.userChildRepo.findOne({
      where: { phoneNumber: dto.loginId },
    });
    if (!child) {
      throw new NotFoundException('Child Not Found!');
    }
    const parent = await this.repo.findOne({ where: { id: child.accountId } });
    if (!parent) {
      throw new NotFoundException('Parent Not Found!');
    }
    const sentOtp = await this.cacheManager.get(dto.loginId);
    if (dto.otp != sentOtp) {
      throw new UnauthorizedException('Invalid otp!');
    }
    const token = await APIFeatures.assignJwtToken(parent.id, this.jwtService);
    return { token, accountId: parent.id, childId: child.id };
  }

  async welcomeMail(dto: EmailDto) {
    try {
      const result = await this.nodeMailerService.userRegisterMail(
        dto.email,
        dto.businessAccId,
      );
      return result
        ? { success: true, message: 'Successfully sent mail' }
        : { success: false, message: 'Failed to send mail!' };
    } catch (error) {
      return { success: false, message: 'Internal error sending mail' };
    }
  }

  async logout(accountId: string, ip: string) {
    const admin = await this.repo
      .createQueryBuilder('account')
      .where('account.id = :id', { id: accountId })
      .getOne();

    const latestLogin = await this.logRepo
      .createQueryBuilder('loginHistory')
      .where(
        'loginHistory.accountId = :accountId AND loginHistory.type = :type',
        {
          accountId: accountId,
          type: LogType.LOGIN,
        },
      )
      .orderBy('loginHistory.createdAt', 'DESC')
      .getOne();
    if (latestLogin) {
      const now = new Date();
      var duration = Math.floor(
        (now.getTime() - new Date(latestLogin.createdAt).getTime()) / 1000,
      ); // in seconds
      // latestLogin.duration = duration;
      // await this.logRepo.save(latestLogin);
    }

    const obj = Object.create({
      loginId: admin.email,
      ip: ip,
      type: LogType.LOGOUT,
      duration: duration,
      accountId: accountId,
    });
    return this.logRepo.save(obj);
  }

  async resetPassword(dto: ForgotPassDto) {
    const user = await this.repo
      .createQueryBuilder('account')
      .where('account.email = :email AND account.roles = :roles', {
        email: dto.email,
        roles: UserRole.ADMIN,
      })
      .getOne();
    if (!user) {
      throw new NotFoundException('Email does not exist!');
    }
    const hashedPassword = await bcrypt.hash(dto.newPassword, 10);
    user.password = hashedPassword;

    await this.repo.save(user);

    return { message: 'Password reset successfully' };
  }

  validate(id: string) {
    return this.getUserDetails(id);
  }

  findPermission(accountId: string) {
    return this.getPermissions(accountId);
  }

  private getPermissions = async (accountId: string): Promise<any> => {
    let result = await this.cacheManager.get('userPermission' + accountId);
    if (!result) {
      result = await this.upRepo.find({
        relations: ['permission', 'menu'],
        where: { accountId, status: true },
      });
      this.cacheManager.set(
        'userPermission' + accountId,
        result,
        7 * 24 * 60 * 60 * 1000,
      );
    }
    return result;
  };

  private getUserDetails = async (
    id: string,
    role?: UserRole,
  ): Promise<any> => {
    const query = this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail');
    if (!role && role == UserRole.USER) {
      query.where('account.roles = :roles', { roles: UserRole.USER });
    }
    if (!role && role == UserRole.BUSINESS) {
      query.where('account.roles IN (:...roles)', {
        roles: [UserRole.BUSINESS],
      });
    }
    if (!role && role == UserRole.ADMIN) {
      query.where('account.roles IN (:...roles)', {
        roles: [UserRole.ADMIN, UserRole.STAFF],
      });
    }
    const result = await query
      .andWhere('account.id = :id OR account.phoneNumber = :phoneNumber', {
        id: id,
        phoneNumber: id,
      })
      .getOne();
    if (!result) {
      throw new UnauthorizedException('Account not found!');
    }
    return result;
  };

  private async getSmsOfBusiness(businessAccId: string) {
    const smsCred = await this.businessRepo.findOne({
      where: { accountId: businessAccId },
    });
    return {
      smsRoute: smsCred.smsRoute,
      smsSenderId: smsCred.smsSenderId,
      smsApiKey: smsCred.smsApiKey,
    };
  }
}
