import {
  Body,
  Controller,
  Get,
  HttpException,
  HttpStatus,
  Ip,
  Post,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AuthService } from './auth.service';
import {
  AdminSigninDto,
  BusinessCreateDto,
  BusinessForgotDto,
  BusinessLoginDto,
  BusinessResetDto,
  EmailDto,
  ForgotPassDto,
  MemberSigninDto,
  OtpDto,
  SigninDto,
  StaffLoginDto,
  VerifyOtpDto,
} from './dto/login.dto';
import { AuthGuard } from '@nestjs/passport';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from './decorators/current-user.decorator';
import { UserRole } from 'src/enum';
import { Roles } from './decorators/roles.decorator';
import { RolesGuard } from './guards/roles.guard';

@Controller('auth')
export class AuthController {
  constructor(private readonly authService: AuthService) {}

  @Post('admin/login')
  adminLogin(@Body() dto: AdminSigninDto) {
    return this.authService.adminLogin(dto);
  }

  @Post('admin/verify')
  verifyOtp(@Body() dto: VerifyOtpDto, @Ip() ip) {
    return this.authService.verifyOtp(dto, ip);
  }

  @Post('admin/resetPass')
  resetPassword(@Body() dto: ForgotPassDto) {
    return this.authService.resetPassword(dto);
  }

  @Post('business/login')
  businessLogin(@Body() dto: BusinessLoginDto) {
    return this.authService.businessLogin(dto);
  }

  @Post('business/verify')
  businessVerifyOTP(@Body() dto: VerifyOtpDto) {
    return this.authService.businessVerifyOTP(dto);
  }

  @Post('staff/login')
  stafflogin(@Body() dto: StaffLoginDto) {
    return this.authService.staffLogin(dto);
  }

  @Post('staff/verify')
  staffVerifyOTP(@Body() dto: VerifyOtpDto) {
    return this.authService.staffVerifyOTP(dto);
  }

  @Post('business/forgot')
  businessforgotPass(@Body() dto: BusinessForgotDto) {
    return this.authService.businessforgotPass(dto);
  }

  @Post('business/otpcheck')
  businessOtpCheck(@Body() dto: VerifyOtpDto) {
    return this.authService.businessOtpCheck(dto.email, dto.otp);
  }

  @Post('business/resetPass')
  businessResetPass(@Body() dto: BusinessResetDto) {
    return this.authService.businessResetPass(dto);
  }

  @Post('member/login')
  memberLogin(@Body() dto: MemberSigninDto) {
    return this.authService.memberLogin(dto);
  }

  @Post('member/verify')
  memberVerifyOtp(@Body() dto: OtpDto) {
    return this.authService.memberVerifyOtp(dto);
  }

  @Post('child/login')
  childLogin(@Body() dto: SigninDto) {
    return this.authService.childLogin(dto);
  }

  @Post('child/verify')
  childVerifyOtp(@Body() dto: OtpDto) {
    return this.authService.childVerifyOtp(dto);
  }

  @Post('register-mail')
  async welcomeMail(@Body() dto: EmailDto) {
    const response = await this.authService.welcomeMail(dto);
    if (!response.success) {
      throw new HttpException(
        response.message,
        HttpStatus.INTERNAL_SERVER_ERROR,
      );
    }
    return response;
  }

  @Get('logout')
  @UseGuards(AuthGuard('jwt'))
  logout(@CurrentUser() user: Account, @Ip() ip) {
    return this.authService.logout(user.id, ip);
  }
}
