import {
  BadRequestException,
  ConflictException,
  Inject,
  Injectable,
  NotFoundException,
  UnauthorizedException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import {
  DefaultStatus,
  MemberStatus,
  SubscriptionStatus,
  UserRole,
} from 'src/enum';
import { Brackets, Repository } from 'typeorm';
import {
  AddMemberDto,
  BusinessCreateByMasterDto,
  ChildQrVerifyDto,
  CreateAccountDto,
  CSVMemberPaginationDto,
  EmailUpdateDto,
  MemberPaginationDto,
  SearchMemberPaginationDto,
  UpdateStaffDto,
  UpdateStaffPasswordDto,
} from './dto/account.dto';
import { Account } from './entities/account.entity';
import { CACHE_MANAGER } from '@nestjs/cache-manager';
import { Cache } from 'cache-manager';
import { UserDetail } from 'src/user-details/entities/user-detail.entity';
import { DefaultStatusDto } from 'src/common/dto/default-status.dto';
import * as bcrypt from 'bcrypt';
import { StaffDetail } from 'src/staff-details/entities/staff-detail.entity';
import { DefaultStatusPaginationDto } from 'src/common/dto/default-status-pagination.dto';
import { createObjectCsvStringifier } from 'csv-writer';
import { Menu } from 'src/menus/entities/menu.entity';
import { MembershipCard } from 'src/membership-card/entities/membership-card.entity';
import { generateQrCode } from 'src/utils/qrCode.util';
import { Business } from 'src/business/entities/business.entity';
import { MenusService } from 'src/menus/menus.service';
import { PermissionsService } from 'src/permissions/permissions.service';
import { UserPermissionsService } from 'src/user-permissions/user-permissions.service';
import { NodeMailerService } from 'src/node-mailer/node-mailer.service';
import { BusinessEmail } from 'src/business-email/entities/business-email.entity';
import { BusinessPayment } from 'src/business-payment/entities/business-payment.entity';

@Injectable()
export class AccountService {
  constructor(
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(UserDetail)
    private readonly udRepo: Repository<UserDetail>,
    @InjectRepository(MembershipCard)
    private readonly memCardRepo: Repository<MembershipCard>,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
    @InjectRepository(StaffDetail)
    private readonly staffRepo: Repository<StaffDetail>,
    @InjectRepository(Menu)
    private readonly menuRepo: Repository<Menu>,
    @InjectRepository(Business)
    private readonly businessRepo: Repository<Business>,
    @InjectRepository(BusinessEmail)
    private readonly bEmailRepo: Repository<BusinessEmail>,
    @InjectRepository(BusinessPayment)
    private readonly bPayRepo: Repository<BusinessPayment>,
    private readonly menuService: MenusService,
    private readonly permissionService: PermissionsService,
    private readonly userPermService: UserPermissionsService,
    private readonly nodemailerService: NodeMailerService,
  ) {}

  async createBusiness(dto: BusinessCreateByMasterDto) {
    const result = await this.repo
      .createQueryBuilder('account')
      .where(
        'account.email = :email OR account.phoneNumber = :phoneNumber AND account.roles = :roles',
        {
          email: dto.email,
          phoneNumber: dto.phoneNumber,
          roles: UserRole.BUSINESS,
        },
      )
      .getOne();
    if (result) {
      throw new ConflictException('Email or Phone Number already exists!');
    }
    const obj = Object.assign({
      email: dto.email,
      phoneNumber: dto.phoneNumber,
      password: await bcrypt.hash(dto.password, 10),
      roles: UserRole.BUSINESS,
    });
    const account = await this.repo.save(obj);

    const busiObj = Object.assign({
      personEmail: dto.email,
      personPhone: dto.phoneNumber,
      accountId: account.id,
    });
    await this.businessRepo.save(busiObj);

    const bEmObj = Object.assign({ accountId: account.id });
    await this.bEmailRepo.save(bEmObj);

    const bPayObj = Object.assign({ accountId: account.id });
    await this.bPayRepo.save(bPayObj);

    const menus = await this.menuService.findAll();
    const perms = await this.permissionService.findAll();
    const permObj = [];
    menus.forEach((menu) => {
      perms.forEach((perm) => {
        permObj.push({
          accountId: account.id,
          menuId: menu.id,
          permissionId: perm.id,
          status: true,
        });
      });
    });
    await this.userPermService.create(permObj);

    return account;
  }

  async create(dto: CreateAccountDto, businessAdminId: string) {
    const user = await this.repo.findOne({
      where: { email: dto.email, roles: UserRole.STAFF },
    });
    if (user) {
      throw new ConflictException('Email already exists!');
    }

    const encryptedPassword = await bcrypt.hash(dto.password, 13);
    const obj = Object.assign({
      email: dto.email,
      password: encryptedPassword,
      businessAdminId: businessAdminId,
      roles: UserRole.STAFF,
      status: dto.status,
    });
    const payload = await this.repo.save(obj);
    const object = Object.assign({
      staffId: '#' + Math.floor(1000 + Math.random() * 9000),
      salutation: dto.salutation,
      gender: dto.gender,
      fName: dto.fName,
      lName: dto.lName,
      mobile: dto.mobile,
      staffRole: dto.staffRole,
      accountId: payload.id,
    });
    await this.staffRepo.save(object);
    return payload;
  }

  async addMember (dto: AddMemberDto, accountId: string) {
    const result = await this.repo.findOne({
      where: { phoneNumber: dto.phoneNumber },
    });
    if (result) {
      throw new ConflictException('Phone number already exists!');
    }
    const membershipCard = await this.memCardRepo.findOne({
      where: { id: dto.membershipCardId },
    });
    if (!membershipCard) {
      throw new NotFoundException('MembershipCard not found!');
    }

    // const today = new Date();
    // const startDate = new Date().toLocaleDateString('en-CA');
    // const duration = parseInt(membershipCard.validity);
    // const endDate = new Date(today);
    // endDate.setDate(today.getDate() + duration - 1);
    // const endDateString = endDate.toLocaleDateString('en-CA');
    // const memberId = `MEM-${Math.floor(1000 + Math.random() * 9000)}`;
    // const cardNumber = `CRD-${Math.floor(1000 + Math.random() * 9000)}`;

    const accObj = Object.create({
      phoneNumber: dto.phoneNumber,
      roles: UserRole.USER,
    });
    const account = await this.repo.save(accObj);
    const udObj = Object.create({
      accountId: account.id,
      createdById: accountId,
      salutation: dto.salutation,
      email: dto.email,
      fName: dto.fName,
      mName: dto.mName,
      lName: dto.lName,
      gender: dto.gender,
      address1: dto.address1,
      address2: dto.address2,
      landMark: dto.landMark,
      city: dto.city,
      state: dto.state,
      zipcode: dto.zipcode,
      aadharNumber: dto.aadharNumber,
      panNumber: dto.panNumber,
      haveBusiness: dto.haveBusiness,
      businessType: dto.businessType,
      businessName: dto.businessName,
      businessEmail: dto.businessEmail,
      gstNumber: dto.gstNumber,
      businessCity: dto.businessCity,
      businessState: dto.businessState,
      businessZipcode: dto.businessZipcode,
      businessPhone: dto.businessPhone,
      businessLandmark: dto.businessLandmark,
      businessTurnover: dto.businessTurnover,
      businessAddress1: dto.businessAddress1,
      businessAddress2: dto.businessAddress2,
      qualification: dto.qualification,
      fatherName: dto.fatherName,
      dob: dto.dob,
      billTo: dto.billTo,
      maritalStatus: dto.maritalStatus,
      income: dto.income,
      profession: dto.profession,

      membershipCardId: dto.membershipCardId,
      cardNumber: null,
      membershipValidFrom: null,
      membershipValidTo: null,
      memberId: null,
    });
    await this.udRepo.save(udObj);
    return account;
  }

  async memberList(dto: MemberPaginationDto, accountId: string) {
    const keyword = dto.keyword || '';
    const startDate = new Date(dto.startDate);
    startDate.setHours(0, 0, 0, 0);
    const endDate = new Date(dto.endDate);
    endDate.setHours(23, 59, 59, 999);

    const joinStartDate = new Date(dto.joinStartDate);
    joinStartDate.setHours(0, 0, 0, 0);
    const joinEndDate = new Date(dto.joinEndDate);
    joinEndDate.setHours(23, 59, 59, 999);

    const query = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        // 'account.status',
        'account.createdAt',

        'userDetail.id',
        'userDetail.memberId',
        'userDetail.oldMemberId',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.email',
        'userDetail.gender',
        'userDetail.profile',
        'userDetail.address1',
        'userDetail.address2',
        'userDetail.city',
        'userDetail.state',
        'userDetail.zipcode',
        'userDetail.memberDoc',
        'userDetail.businessType',
        'userDetail.businessName',
        'userDetail.businessTurnover',
        'userDetail.businessEmail',
        'userDetail.gstNumber',
        'userDetail.businessDoc',
        'userDetail.businessCity',
        'userDetail.businessState',
        'userDetail.businessZipcode',
        'userDetail.businessPhone',
        'userDetail.businessLandmark',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.cardNumber',
        'userDetail.landMark',
        'userDetail.fatherName',
        'userDetail.dob',
        'userDetail.qualification',
        'userDetail.profession',
        'userDetail.panNumber',
        'userDetail.income',
        'userDetail.salutation',
        'userDetail.haveBusiness',
        'userDetail.businessAddress1',
        'userDetail.businessAddress2',
        'userDetail.pan',
        'userDetail.aadhar',
        'userDetail.aadharNumber',
        'userDetail.status',
        'userDetail.paymentStatus',
        'userDetail.createdById',
        'userDetail.maritalStatus',
        'userDetail.billTo',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',
        'userDetail.formNum',

        'membershipCard.id',
        'membershipCard.businessName',
        'membershipCard.name',
        'membershipCard.validity',
        'membershipCard.price',
        'membershipCard.currencyType',
        'membershipCard.memberCount',
      ])
      .where(
        'account.roles = :roles AND userDetail.createdById = :createdById',
        { roles: UserRole.USER, createdById: accountId },
      );
    if (dto.status && dto.status.length > 0) {
      query.andWhere('userDetail.status = :status', {
        status: dto.status,
      });
    }
    if (dto.paymentStatus && dto.paymentStatus.length > 0) {
      query.andWhere('userDetail.paymentStatus = :paymentStatus', {
        paymentStatus: dto.paymentStatus,
      });
    }
    if (dto.phoneNumber && dto.phoneNumber.length > 0) {
      query.andWhere('account.phoneNumber = :phoneNumber', {
        phoneNumber: dto.phoneNumber,
      });
    }
    if (dto.membershipType && dto.membershipType.length > 0) {
      query.andWhere('membershipCard.name = :name', {
        name: dto.membershipType,
      });
    }
    if (dto.memberId && dto.memberId.length > 0) {
      query.andWhere('userDetail.memberId LIKE :memberId', {
        memberId: `%${dto.memberId}%`,
      });
    }
    if (dto.startDate && dto.endDate) {
      query.andWhere(
        'userDetail.membershipValidTo >= :startDate AND userDetail.membershipValidTo <= :endDate',
        {
          startDate: startDate,
          endDate: endDate,
        },
      );
    }
    if (dto.joinStartDate && dto.joinEndDate) {
      query.andWhere(
        'account.createdAt >= :joinStartDate AND account.createdAt <= :joinEndDate',
        {
          joinStartDate: joinStartDate,
          joinEndDate: joinEndDate,
        },
      );
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where(
            'userDetail.fName LIKE :keyword OR userDetail.mName LIKE :keyword OR userDetail.lName LIKE :keyword OR userDetail.gender LIKE :keyword OR userDetail.address1 LIKE :keyword OR userDetail.address2 LIKE :keyword OR userDetail.businessName LIKE :keyword OR userDetail.gstNumber LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      );
    }
    const [result, total] = await query
      .orderBy({ 'account.createdAt': 'DESC' })
      .take(dto.limit)
      .skip(dto.offset)
      .getManyAndCount();

    return { result, total };
  }

  async memberListforCsv(dto: CSVMemberPaginationDto, accountId: string) {
    const keyword = dto.keyword || '';
    const startDate = new Date(dto.startDate);
    startDate.setHours(0, 0, 0, 0);
    const endDate = new Date(dto.endDate);
    endDate.setHours(23, 59, 59, 999);

    const joinStartDate = new Date(dto.joinStartDate);
    joinStartDate.setHours(0, 0, 0, 0);
    const joinEndDate = new Date(dto.joinEndDate);
    joinEndDate.setHours(23, 59, 59, 999);

    const query = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        // 'account.status',
        'account.createdAt',

        'userDetail.id',
        'userDetail.formNum',
        'userDetail.memberId',
        'userDetail.oldMemberId',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.email',
        'userDetail.gender',
        'userDetail.profile',
        'userDetail.address1',
        'userDetail.address2',
        'userDetail.city',
        'userDetail.state',
        'userDetail.zipcode',
        'userDetail.memberDoc',
        'userDetail.businessType',
        'userDetail.businessName',
        'userDetail.businessTurnover',
        'userDetail.businessEmail',
        'userDetail.gstNumber',
        'userDetail.businessDoc',
        'userDetail.businessCity',
        'userDetail.businessState',
        'userDetail.businessZipcode',
        'userDetail.businessPhone',
        'userDetail.businessLandmark',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.cardNumber',
        'userDetail.landMark',
        'userDetail.fatherName',
        'userDetail.dob',
        'userDetail.qualification',
        'userDetail.profession',
        'userDetail.panNumber',
        'userDetail.income',
        'userDetail.salutation',
        'userDetail.haveBusiness',
        'userDetail.businessAddress1',
        'userDetail.businessAddress2',
        'userDetail.pan',
        'userDetail.aadhar',
        'userDetail.aadharNumber',
        'userDetail.status',
        'userDetail.paymentStatus',
        'userDetail.createdById',
        'userDetail.maritalStatus',
        'userDetail.billTo',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',

        'membershipCard.id',
        'membershipCard.businessName',
        'membershipCard.name',
        'membershipCard.validity',
        'membershipCard.price',
        'membershipCard.currencyType',
        'membershipCard.memberCount',
      ])
      .where(
        'account.roles = :roles AND userDetail.createdById = :createdById',
        { roles: UserRole.USER, createdById: accountId },
      );
    if (dto.status && dto.status.length > 0) {
      query.andWhere('userDetail.status = :status', {
        status: dto.status,
      });
    }
    if (dto.paymentStatus && dto.paymentStatus.length > 0) {
      query.andWhere('userDetail.paymentStatus = :paymentStatus', {
        paymentStatus: dto.paymentStatus,
      });
    }
    if (dto.phoneNumber && dto.phoneNumber.length > 0) {
      query.andWhere('account.phoneNumber = :phoneNumber', {
        phoneNumber: dto.phoneNumber,
      });
    }
    if (dto.membershipType && dto.membershipType.length > 0) {
      query.andWhere('membershipCard.name = :name', {
        name: dto.membershipType,
      });
    }
    if (dto.memberId && dto.memberId.length > 0) {
      query.andWhere('userDetail.memberId LIKE :memberId', {
        memberId: `%${dto.memberId}%`,
      });
    }
    if (dto.startDate && dto.endDate) {
      query.andWhere(
        'userDetail.membershipValidTo >= :startDate AND userDetail.membershipValidTo <= :endDate',
        {
          startDate: startDate,
          endDate: endDate,
        },
      );
    }
    if (dto.joinStartDate && dto.joinEndDate) {
      query.andWhere(
        'account.createdAt >= :joinStartDate AND account.createdAt <= :joinEndDate',
        {
          joinStartDate: joinStartDate,
          joinEndDate: joinEndDate,
        },
      );
    }
    if (dto.keyword && dto.keyword.length > 0) {
      query.andWhere(
        new Brackets((qb) => {
          qb.where(
            'userDetail.fName LIKE :keyword OR userDetail.mName LIKE :keyword OR userDetail.lName LIKE :keyword OR userDetail.gender LIKE :keyword OR userDetail.address1 LIKE :keyword OR userDetail.address2 LIKE :keyword OR userDetail.businessName LIKE :keyword OR userDetail.gstNumber LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      );
    }
    const [result, total] = await query
      .orderBy({ 'account.createdAt': 'DESC' })
      .getManyAndCount();

    return { result, total };
  }

  async downloadMemberCSV(dto: CSVMemberPaginationDto, accountId: string) {
    const formatDate = (dateString) => {
      const date = new Date(dateString);
      const year = date.getFullYear();
      const month = String(date.getMonth() + 1).padStart(2, '0');
      const day = String(date.getDate()).padStart(2, '0');
      return `${day}-${month}-${year}`;
    };

    const { result } = await this.memberListforCsv(dto, accountId);

    const csvStringifier = createObjectCsvStringifier({
      header: [
        // { id: 'createdAt', title: 'Created At' },
        { id: 'memberId', title: 'Member Id' },
        { id: 'phoneNumber', title: 'Phone Number' },
        { id: 'salutation', title: 'Salutation' },
        { id: 'userName', title: 'User Name' },
        { id: 'email', title: 'Email' },
        { id: 'gender', title: 'Gender' },
        { id: 'maritalStatus', title: 'Marital Status' },
        { id: 'dob', title: 'Date of Birth' },
        { id: 'fatherName', title: "Father's Name" },
        { id: 'address1', title: 'Address 1' },
        { id: 'address2', title: 'Address 2' },
        { id: 'landMark', title: 'Landmark' },
        { id: 'qualification', title: 'Qualification' },
        { id: 'profession', title: 'Profession' },
        { id: 'panNumber', title: 'Pan Number' },
        { id: 'state', title: 'State' },
        { id: 'city', title: 'City' },
        { id: 'zipcode', title: 'Pincode' },
        { id: 'profile', title: 'Profile URL' },
        { id: 'income', title: 'Income' },
        { id: 'memberDoc', title: 'Document URL' },
        { id: 'billTo', title: 'Bill To' },
        { id: 'membershipValidFrom', title: 'Membership From' },
        { id: 'membershipValidTo', title: 'Membership To' },
        { id: 'cardNumber', title: 'Card Number' },
        { id: 'status', title: 'Status' },
        { id: 'paymentStatus', title: 'Payment Status' },

        // Separate fields for if have business
        { id: 'businessName', title: 'Business Name' },
        { id: 'businessType', title: 'Business Type' },
        { id: 'gstNumber', title: 'Gst Number' },
        { id: 'businessDoc', title: 'Business Document URL' },
        { id: 'businessCity', title: 'Business City' },
        { id: 'businessState', title: 'Business State' },
        { id: 'businessZipcode', title: 'Business Pincode' },
        { id: 'businessEmail', title: 'Business Email' },
        { id: 'businessPhone', title: 'Business Phone' },
        { id: 'businessAddress1', title: 'Business Address 1' },
        { id: 'businessAddress2', title: 'Business Address 2' },
        { id: 'businessTurnover', title: 'Business Turnover' },

        // Separate fields for Membership Card
        { id: 'membershipCardName', title: 'Membership' },
        { id: 'price', title: 'Card Price' },
        { id: 'dependentCount', title: 'Dependent Count' },
      ],
    });

    const records = result.map((account) => {
      const userDetail = account.userDetail[0];

      const baseRecord = {
        // createdAt: formatDate(account.createdAt),
        memberId: userDetail.memberId,
        phoneNumber: account.phoneNumber,
        salutation: userDetail.salutation,
        userName: `${userDetail.fName} ${userDetail.mName ? `${userDetail.mName} ${userDetail.lName}` : userDetail.lName}`,
        email: userDetail.email,
        gender: userDetail.gender,
        maritalStatus: userDetail.maritalStatus,
        dob: userDetail.dob,
        fatherName: userDetail.fatherName,
        address1: userDetail.address1,
        address2: userDetail.address2,
        landMark: userDetail.landMark,
        qualification: userDetail.qualification,
        profession: userDetail.profession,
        panNumber: userDetail.panNumber,
        state: userDetail.state,
        city: userDetail.city,
        zipcode: userDetail.zipcode,
        profile: userDetail.profile,
        income: userDetail.income,
        memberDoc: userDetail.memberDoc,
        billTo: userDetail.billTo,
        membershipValidFrom: userDetail.membershipValidFrom,
        membershipValidTo: userDetail.membershipValidTo,
        cardNumber: userDetail.cardNumber,
        status: userDetail.status,
        paymentStatus: userDetail.paymentStatus,

        //membership card details
        // membershipCardName: userDetail.membershipCard['name'],
        // price: userDetail.membershipCard['price'],
        // dependentCount: userDetail.membershipCard['memberCount'],
      };

      if (userDetail.haveBusiness === 'Yes') {
        baseRecord['businessName'] = userDetail.businessName;
        baseRecord['businessType'] = userDetail.businessType;
        baseRecord['gstNumber'] = userDetail.gstNumber;
        baseRecord['businessDoc'] = userDetail.businessDoc;
        baseRecord['businessCity'] = userDetail.businessCity;
        baseRecord['businessState'] = userDetail.businessState;
        baseRecord['businessZipcode'] = userDetail.businessZipcode;
        baseRecord['businessEmail'] = userDetail.businessEmail;
        baseRecord['businessPhone'] = userDetail.businessPhone;
        baseRecord['businessAddress1'] = userDetail.businessAddress1;
        baseRecord['businessAddress2'] = userDetail.businessAddress2;
        baseRecord['businessTurnover'] = userDetail.businessTurnover;
      } else {
        baseRecord['businessName'] = '';
        baseRecord['businessType'] = '';
        baseRecord['gstNumber'] = '';
        baseRecord['businessDoc'] = '';
        baseRecord['businessCity'] = '';
        baseRecord['businessState'] = '';
        baseRecord['businessZipcode'] = '';
        baseRecord['businessEmail'] = '';
        baseRecord['businessPhone'] = '';
        baseRecord['businessAddress1'] = '';
        baseRecord['businessAddress2'] = '';
        baseRecord['businessTurnover'] = '';
      }

      if (userDetail.membershipCard == null || "") {
        baseRecord['membershipCardName'] = '';
        baseRecord['price'] = '';
        baseRecord['dependentCount'] = '';
      } else {
        baseRecord['membershipCardName'] = userDetail.membershipCard['name'];
        baseRecord['price'] = userDetail.membershipCard['price'];
        baseRecord['dependentCount'] = userDetail.membershipCard['memberCount'];
      }
      return baseRecord;
    });

    const csvContent =
      csvStringifier.getHeaderString() +
      csvStringifier.stringifyRecords(records);

    return csvContent;
  }

  async findOneMember(accountId: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .leftJoinAndSelect('membershipCard.cardAmenities', 'cardAmenities')
      .leftJoinAndSelect('account.userChild', 'userChild')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',

        'userDetail.id',
        'userDetail.formNum',
        'userDetail.createdById',
        'userDetail.memberId',
        'userDetail.oldMemberId',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.email',
        'userDetail.gender',
        'userDetail.profile',
        'userDetail.address1',
        'userDetail.address2',
        'userDetail.city',
        'userDetail.state',
        'userDetail.zipcode',
        'userDetail.memberDoc',
        'userDetail.businessType',
        'userDetail.businessName',
        'userDetail.businessTurnover',
        'userDetail.businessEmail',
        'userDetail.gstNumber',
        'userDetail.businessDoc',
        'userDetail.businessCity',
        'userDetail.businessState',
        'userDetail.businessZipcode',
        'userDetail.businessPhone',
        'userDetail.cardNumber',
        'userDetail.landMark',
        'userDetail.businessLandmark',
        'userDetail.fatherName',
        'userDetail.dob',
        'userDetail.qualification',
        'userDetail.profession',
        'userDetail.panNumber',
        'userDetail.income',
        'userDetail.salutation',
        'userDetail.haveBusiness',
        'userDetail.businessAddress1',
        'userDetail.businessAddress2',
        'userDetail.pan',
        'userDetail.aadhar',
        'userDetail.aadharNumber',
        'userDetail.status',
        'userDetail.paymentStatus',
        'userDetail.maritalStatus',
        'userDetail.billTo',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.validity',
        'membershipCard.price',
        'membershipCard.currencyType',
        'membershipCard.memberCount',
        'membershipCard.cardDesign',

        'cardAmenities.id',
        'cardAmenities.name',
        'cardAmenities.icon',
        'cardAmenities.desc',
        'cardAmenities.shortDesc',

        'userChild.id',
        'userChild.memberId',
        'userChild.oldDepMemberId',
        'userChild.childCardNumber',
        'userChild.name',
        'userChild.email',
        'userChild.phoneNumber',
        'userChild.relation',
        'userChild.martialStatus',
        'userChild.dob',
        'userChild.doc',
        'userChild.profile',
        'userChild.createAt',
        'userChild.updatedAt',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    if (!result) {
      throw new NotFoundException('Member not found!');
    }
    return result;
  }

  async findBySearch(dto: SearchMemberPaginationDto) {
    const keyword = dto.keyword || '';
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      // .leftJoinAndSelect('account.userChild', 'userChild')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',

        'userDetail.id',
        'userDetail.formNum',
        'userDetail.createdById',
        'userDetail.memberId',
        'userDetail.status',
        'userDetail.paymentStatus',

        // 'userChild.id',
        // 'userChild.memberId',
        // 'userChild.childCardNumber',
        // 'userChild.phoneNumber',
      ])
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.phoneNumber LIKE :keyword OR userDetail.memberId LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .getOne();
    if (!result) {
      throw new NotFoundException('Member Not Found!!');
    }
    return result;
  }

  async businessProfile(accountId: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .leftJoinAndSelect('business.licence', 'licence')
      .leftJoinAndSelect('account.setting', 'setting')
      .leftJoinAndSelect('account.tax', 'tax')
      .leftJoinAndSelect('account.membershipCard', 'membershipCard')
      .leftJoinAndSelect('account.businessPayment', 'businessPayment')
      .leftJoinAndSelect('account.businessEmail', 'businessEmail')
      .select([
        'account.id',
        'account.email',
        'account.phoneNumber',
        'account.roles',
        'account.createdAt',

        'business.id',
        'business.gender',
        'business.personName',
        'business.personDesignation',
        'business.personEmail',
        'business.personPhone',
        'business.businessKey',
        'business.businessType',
        'business.businessName',
        'business.parentCompanyName',
        'business.businessPhone',
        'business.businessEmail',
        'business.gstNo',
        'business.address1',
        'business.address2',
        'business.zipCode',
        'business.city',
        'business.state',
        'business.country',
        'business.signatory',
        'business.logo',
        'business.brandLogo',
        'business.doc1',
        'business.doc2',
        'business.gstCertificate',
        'business.workOrder',
        'business.status',
        'business.accountId',
        'business.createdAt',
        'business.updatedAt',
        'business.smsRoute',
        'business.smsSenderId',
        'business.smsApiKey',

        'licence.id',
        'licence.businessId',
        'licence.userLimit',
        'licence.licenceKey',
        'licence.activationKey',
        'licence.startDate',
        'licence.renewalDate',
        'licence.amc',
        'licence.createdAt',
        'licence.status',

        'setting.id',
        'setting.title',
        'setting.user_domain',
        'setting.admin_domain',
        'setting.mobile_domain',
        'setting.dateFormat',
        'setting.timeFormat',
        'setting.timeZone',
        'setting.currency',
        'setting.createdAt',

        'tax.id',
        'tax.taxName',
        'tax.rate',
        'tax.status',
        'tax.createdAt',
        'tax.updatedAt',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.validity',
        'membershipCard.price',
        'membershipCard.currencyType',
        'membershipCard.cardType',
        'membershipCard.desc',
        'membershipCard.memberCount',
        'membershipCard.status',
        'membershipCard.cardDesign',
        'membershipCard.createdAt',

        'businessPayment.id',
        'businessPayment.keyId',
        'businessPayment.keySecret',
        'businessPayment.gatewayName',
        'businessPayment.status',

        'businessEmail.id',
        'businessEmail.mailEngine',
        'businessEmail.smtpHost',
        'businessEmail.smtpPort',
        'businessEmail.username',
        'businessEmail.password',
        'businessEmail.from',
        'businessEmail.secure',
        'businessEmail.status',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    return result;
  }

  async generateBusinessQR(accountId: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .select(['account.id', 'business.businessName'])
      .where('account.id = :id', { id: accountId })
      .getOne();
    // return result;
    const qrCode = await generateQrCode(result);
    return { qrCode: qrCode };
  }

  async verifyUserByQR(businessAccId: string, accountId: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',

        'userDetail.id',
        'userDetail.memberId',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.status',
        'userDetail.paymentStatus',

        'membershipCard.id',
        'membershipCard.name',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    if (!result || !result.userDetail) {
      throw new NotFoundException('User not found');
    }

    const business = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .leftJoinAndSelect(
        'account.membershipCard',
        'membershipCard',
        'membershipCard.status = :status',
        {
          status: DefaultStatus.ACTIVE,
        },
      )
      .select([
        'account.id',

        'business.businessName',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.status',
      ])
      .where('account.id = :id', { id: businessAccId })
      .getOne();
    if (!business) {
      throw new NotFoundException('Business account not found');
    }

    if (result.userDetail[0].paymentStatus != SubscriptionStatus.COMPLETED) {
      return {
        message: 'User Authenticated.',
        authenticated: true,
        renewal: false,
      };
    }

    const userCardId = result.userDetail[0].membershipCard['id'];
    if (!userCardId) {
      throw new NotFoundException('User does not have a valid membership card');
    }

    const isCardValid = business.membershipCard?.some(
      (card) => card.id === userCardId,
    );
    if (isCardValid) {
      return {
        message: 'User Authenticated.',
        authenticated: true,
        renewal: true,
        fName: result.userDetail[0].fName,
        mName: result.userDetail[0].mName,
        lName: result.userDetail[0].lName,
        cardName: result.userDetail[0].membershipCard['name'],
      };
    } else {
      throw new NotFoundException('User Not found or invalid membership');
    }
  }

  async verifyChildByQR(dto: ChildQrVerifyDto) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .leftJoinAndSelect('account.userChild', 'userChild')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',

        'userDetail.id',
        'userDetail.memberId',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.status',
        'userDetail.paymentStatus',

        'membershipCard.id',
        'membershipCard.name',

        'userChild.id',
        'userChild.memberId',
        'userChild.childCardNumber',
        'userChild.name',
        'userChild.email',
        'userChild.phoneNumber',
        'userChild.relation',
        'userChild.martialStatus',
        'userChild.profile',
        'userChild.createAt',
        'userChild.updatedAt',
      ])
      .where('userChild.id = :id', { id: dto.userChildId })
      .getOne();
    if (!result || !result.userChild) {
      throw new NotFoundException('Child not found');
    }

    const business = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .leftJoinAndSelect(
        'account.membershipCard',
        'membershipCard',
        'membershipCard.status = :status',
        {
          status: DefaultStatus.ACTIVE,
        },
      )
      .select([
        'account.id',

        'business.businessName',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.status',
      ])
      .where('account.id = :id', { id: dto.businessAccId })
      .getOne();
    if (!business) {
      throw new NotFoundException('Business account not found');
    }
    if (result.userDetail[0].paymentStatus != SubscriptionStatus.COMPLETED) {
      return { renewal: false };
    }
    const userCardId = result.userDetail[0].membershipCard['id'];
    if (!userCardId) {
      throw new NotFoundException('User does not have a valid membership card');
    }

    const isCardValid = business.membershipCard?.some(
      (card) => card.id === userCardId,
    );
    if (isCardValid) {
      return { message: 'Child Authenticated.', authenticated: true };
    } else {
      throw new NotFoundException('User Not found or invalid membership');
    }
  }

  async verifyUserByQRByBusiness(accountId: string, userAccId: string) {
    const business = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .leftJoinAndSelect(
        'account.membershipCard',
        'membershipCard',
        'membershipCard.status = :status',
        {
          status: DefaultStatus.ACTIVE,
        },
      )
      .select([
        'account.id',

        'business.businessName',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.status',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    if (!business) {
      throw new NotFoundException('Business account not found');
    }

    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',

        'userDetail.id',
        'userDetail.memberId',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.status',
        'userDetail.paymentStatus',

        'membershipCard.id',
        'membershipCard.name',
      ])
      .where('account.id = :id', { id: userAccId })
      .getOne();
    if (!result || !result.userDetail) {
      throw new NotFoundException('User not found');
    }

    if (result.userDetail[0].paymentStatus != SubscriptionStatus.COMPLETED) {
      throw new NotFoundException(`Please renew or buy a membership plan!`);
    }

    const userCardId = result.userDetail[0].membershipCard['id'];
    if (!userCardId) {
      throw new NotFoundException('User does not have a valid membership card');
    }

    const isCardValid = business.membershipCard?.some(
      (card) => card.id === userCardId,
    );
    if (isCardValid) {
      return {
        id: result.id,
        userDetailId: result.userDetail[0].id,
        authenticated: true,
        role: 'USER',
        message: 'User Authenticated.',
      };
    } else {
      throw new NotFoundException('User Not found or invalid membership');
    }
  }

  async verifyChildQRByBusiness(accountId: string, userChildId: string) {
    const business = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.business', 'business')
      .leftJoinAndSelect(
        'account.membershipCard',
        'membershipCard',
        'membershipCard.status = :status',
        {
          status: DefaultStatus.ACTIVE,
        },
      )
      .select([
        'account.id',

        'business.businessName',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.status',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    if (!business) {
      throw new NotFoundException('Business account not found');
    }

    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .leftJoinAndSelect('account.userChild', 'userChild')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',

        'userDetail.id',
        'userDetail.memberId',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.status',
        'userDetail.paymentStatus',

        'membershipCard.id',
        'membershipCard.name',

        'userChild.id',
        'userChild.memberId',
        'userChild.childCardNumber',
        'userChild.name',
        'userChild.email',
        'userChild.phoneNumber',
        'userChild.relation',
        'userChild.martialStatus',
        'userChild.profile',
        'userChild.createAt',
        'userChild.updatedAt',
      ])
      .where('userChild.id = :id', { id: userChildId })
      .getOne();
    if (!result || !result.userChild) {
      throw new NotFoundException('Child not found');
    }

    if (result.userDetail[0].paymentStatus != SubscriptionStatus.COMPLETED) {
      throw new UnauthorizedException(`Please renew or buy a membership plan!`);
    }

    const userCardId = result.userDetail[0].membershipCard['id'];
    if (!userCardId) {
      throw new NotFoundException(
        'Parent does not have a valid membership card',
      );
    }

    const isCardValid = business.membershipCard?.some(
      (card) => card.id === userCardId,
    );
    if (isCardValid) {
      return {
        parentId: result.id,
        userChildId: userChildId,
        role: 'CHILD',
        message: 'Child Authenticated.',
      };
    } else {
      throw new NotFoundException('Child Not found or invalid membership');
    }
  }

  async adminProfile(accountId: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.adminDetail', 'adminDetail')
      .leftJoinAndSelect('account.businessEmail', 'businessEmail')
      .select([
        'account.id',
        'account.email',
        'account.password',
        'account.roles',
        'account.status',
        'account.createdAt',

        'adminDetail.id',
        'adminDetail.accountId',
        'adminDetail.userName',
        'adminDetail.phoneNumber',
        'adminDetail.profileImage',
        'adminDetail.smsRoute',
        'adminDetail.smsSenderId',
        'adminDetail.smsApiKey',

        'businessEmail.id',
        'businessEmail.mailEngine',
        'businessEmail.smtpHost',
        'businessEmail.smtpPort',
        'businessEmail.username',
        'businessEmail.password',
        'businessEmail.from',
        'businessEmail.secure',
        'businessEmail.status',
      ])
      .where('account.id = :id AND account.roles = :roles', {
        id: accountId,
        roles: UserRole.ADMIN,
      })
      .getOne();
    return result;
  }

  async getStaffDetails(
    dto: DefaultStatusPaginationDto,
    businessAdminId: string,
  ) {
    const keyword = dto.keyword || '';
    const query = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.businessAdminId',
        'account.email',
        'account.password',
        'account.roles',
        'account.status',
        'account.createdAt',

        'staffDetail.id',
        'staffDetail.staffId',
        'staffDetail.salutation',
        'staffDetail.gender',
        'staffDetail.fName',
        'staffDetail.lName',
        'staffDetail.mobile',
        'staffDetail.staffRole',
        'staffDetail.createdAt',
      ])
      .where(
        'account.roles = :roles AND account.status = :status AND account.businessAdminId = :businessAdminId',
        {
          roles: UserRole.STAFF,
          status: dto.status,
          businessAdminId: businessAdminId,
        },
      );

    const [result, total] = await query
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.phoneNumber LIKE :keyword OR staffDetail.fName LIKE :keyword OR staffDetail.lName LIKE :keyword',
            {
              keyword: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'account.createdAt': 'DESC' })
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  async getStaffProfile(accountId: string) {
    const staff = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.email',
        'account.businessAdminId',
        'account.roles',
        'account.status',

        'staffDetail.id',
        'staffDetail.staffId',
        'staffDetail.salutation',
        'staffDetail.gender',
        'staffDetail.fName',
        'staffDetail.lName',
        'staffDetail.mobile',
        'staffDetail.staffRole',
        'staffDetail.createdAt',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();

    const perms = await this.menuRepo
      .createQueryBuilder('menu')
      .leftJoinAndSelect('menu.userPermission', 'userPermission')
      .leftJoinAndSelect('userPermission.permission', 'permission')
      .where('userPermission.accountId = :accountId', {
        accountId: accountId,
      })
      .orderBy({ 'menu.title': 'ASC', 'permission.id': 'ASC' })
      .getMany();

    return { staff: staff, perms };
  }

  async userProfile(accountId: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      .leftJoinAndSelect('account.userChild', 'userChild')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.status',
        'account.createdAt',

        'userDetail.id',
        'userDetail.formNum',
        'userDetail.memberId',
        'userDetail.oldMemberId',
        'userDetail.membershipValidFrom',
        'userDetail.membershipValidTo',
        'userDetail.fName',
        'userDetail.mName',
        'userDetail.lName',
        'userDetail.email',
        'userDetail.gender',
        'userDetail.profile',
        'userDetail.address1',
        'userDetail.address2',
        'userDetail.city',
        'userDetail.state',
        'userDetail.zipcode',
        'userDetail.memberDoc',
        'userDetail.businessType',
        'userDetail.businessName',
        'userDetail.businessTurnover',
        'userDetail.businessEmail',
        'userDetail.gstNumber',
        'userDetail.businessDoc',
        'userDetail.businessCity',
        'userDetail.businessState',
        'userDetail.businessZipcode',
        'userDetail.businessPhone',
        'userDetail.businessEmail',
        'userDetail.cardNumber',
        'userDetail.landMark',
        'userDetail.businessLandmark',
        'userDetail.fatherName',
        'userDetail.dob',
        'userDetail.qualification',
        'userDetail.profession',
        'userDetail.panNumber',
        'userDetail.income',
        'userDetail.salutation',
        'userDetail.haveBusiness',
        'userDetail.businessAddress1',
        'userDetail.businessAddress2',
        'userDetail.pan',
        'userDetail.aadhar',
        'userDetail.aadharNumber',
        'userDetail.status',
        'userDetail.paymentStatus',
        'userDetail.maritalStatus',
        'userDetail.billTo',
        'userDetail.remarks',
        'userDetail.adminFile',
        'userDetail.regFile',

        'membershipCard.id',
        'membershipCard.name',
        'membershipCard.validity',
        'membershipCard.price',
        'membershipCard.currencyType',
        'membershipCard.memberCount',
        'membershipCard.cardDesign',

        'userChild.id',
        'userChild.memberId',
        'userChild.oldDepMemberId',
        'userChild.childCardNumber',
        'userChild.name',
        'userChild.email',
        'userChild.phoneNumber',
        'userChild.relation',
        'userChild.martialStatus',
        'userChild.dob',
        'userChild.doc',
        'userChild.profile',
        'userChild.createAt',
        'userChild.updatedAt',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    return result;
  }

  async userDetailQRCode(accountId: string) {
    const userProfile = await this.repo
      .createQueryBuilder('account')
      // .leftJoinAndSelect('account.userDetail', 'userDetail')
      // .leftJoinAndSelect('userDetail.membershipCard', 'membershipCard')
      // .leftJoinAndSelect('account.userChild', 'userChild')
      .select([
        'account.id',
        'account.phoneNumber',
        'account.roles',
        'account.createdAt',

        // 'userDetail.id',
        // 'userDetail.memberId',
        // 'userDetail.membershipValidFrom',
        // 'userDetail.membershipValidTo',
        // 'userDetail.fName',
        // 'userDetail.mName',
        // 'userDetail.lName',
        // 'userDetail.email',
        // 'userDetail.gender',
        // 'userDetail.profile',
        // 'userDetail.address1',
        // 'userDetail.address2',
        // 'userDetail.city',
        // 'userDetail.state',
        // 'userDetail.zipcode',
        // 'userDetail.businessType',
        // 'userDetail.businessName',
        // 'userDetail.businessEmail',
        // 'userDetail.memberDoc',
        // 'userDetail.gstNumber',
        // 'userDetail.businessDoc',
        // 'userDetail.businessCity',
        // 'userDetail.businessState',
        // 'userDetail.businessZipcode',
        // 'userDetail.businessPhone',
        // 'userDetail.cardNumber',
        // 'userDetail.landMark',
        // 'userDetail.fatherName',
        // 'userDetail.dob',
        // 'userDetail.qualification',
        // 'userDetail.profession',
        // 'userDetail.panNumber',
        // 'userDetail.income',
        // 'userDetail.status',
        // 'userDetail.maritalStatus',
        // 'userDetail.billTo',

        // 'membershipCard.id',
        // 'membershipCard.name',
        // 'membershipCard.validity',
        // 'membershipCard.price',
        // 'membershipCard.currencyType',
        // 'membershipCard.memberCount',
        // 'membershipCard.cardDesign',

        // 'userChild.id',
        // 'userChild.memberId',
        // 'userChild.childCardNumber',
        // 'userChild.name',
        // 'userChild.email',
        // 'userChild.phoneNumber',
        // 'userChild.relation',
        // 'userChild.martialStatus',
        // 'userChild.profile',
        // 'userChild.createAt',
        // 'userChild.updatedAt',
      ])
      .where('account.id = :id', { id: accountId })
      .getOne();
    if (!userProfile) {
      throw new Error('User not found');
    }
    const qrCode = await generateQrCode(userProfile);
    return { qrCode: qrCode };
  }

  async resetEmail(dto: EmailUpdateDto, accountId: string) {
    const storedOtp = await this.cacheManager.get<string>(dto.email);
    if (storedOtp !== dto.otp) {
      throw new BadRequestException('Invalid or expired OTP');
    }
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.userDetail', 'userDetail')
      .select([
        'account.id',
        'account.email',
        'account.roles',
        'account.status',

        'userDetail.id',
        'userDetail.email',
      ])
      .where(
        'account.id = :id AND account.roles = :roles AND account.status = :status',
        {
          id: accountId,
          roles: UserRole.USER,
          status: DefaultStatus.ACTIVE,
        },
      )
      .getOne();
    const obj = Object.assign(result, { email: dto.email });
    const account = await this.repo.save(obj);

    result.userDetail[0].email = dto.email;
    this.udRepo.save(result.userDetail[0]);
    return account;
  }

  async updateStaff(accountId: string, dto: UpdateStaffDto) {
    const result = await this.staffRepo.findOne({ where: { accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    const obj = Object.assign(result, dto);
    return this.staffRepo.save(obj);
  }

  async updateStaffPassword(accountId: string, dto: UpdateStaffPasswordDto) {
    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    if (dto.loginId && dto.loginId.length > 0) {
      const obj = Object.assign(result, { phoneNumber: dto.loginId });
      await this.repo.save(obj);
    }
    if (dto.password && dto.password.length > 0) {
      const password = await bcrypt.hash(dto.password, 10);
      const obj = Object.assign(result, { password: password });
      await this.repo.save(obj);
    }
    return result;
  }

  async staffStatus(id: string, dto: DefaultStatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Account Not Found With This ID.');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async deleteMember (accountId: string){
    const result = await this.repo.findOne({ where: { id: accountId } });
    if (!result) {
      throw new NotFoundException('Account Not Found!');
    }
    return this.repo.remove(result);
  }
}
