import {
  Body,
  Controller,
  FileTypeValidator,
  Get,
  MaxFileSizeValidator,
  Param,
  ParseFilePipe,
  Patch,
  Post,
  Query,
  UploadedFile,
  UseGuards,
  UseInterceptors,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { FileInterceptor } from '@nestjs/platform-express';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { NotifyService } from 'src/notify/notify.service';
import { PaginationAllDto, TicketAnswerDto, TicketDto, PaginationReportDto } from './dto/ticket.dto';
import { TicketsService } from './tickets.service';

@Controller('tickets')
export class TicketsController {
  constructor(
    private readonly ticketsService: TicketsService,
    private readonly notifyService: NotifyService,
  ) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @UseInterceptors(FileInterceptor('file'))
  create(
    @Body() dto: TicketDto,
    @CurrentUser() user: Account,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new FileTypeValidator({ fileType: '.(png|jpeg|jpg)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    dto.accountId = user.id;
    dto.settingId = user.settingId;
    dto.image = process.env.CDN_LINK + file.path;
    dto.imageName = file.path;
    this.notifyService.raiseTicket(user.fcm, user.id);
    return this.ticketsService.create(dto);
  }

  @Get('all')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'ticket'])
  findAll(@Query() dto: PaginationAllDto) {
    return this.ticketsService.findAll(dto);
  }

  @Get('/report')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'ticket'])
  findReportAll(@Query() dto: PaginationReportDto) {
    return this.ticketsService.findReportAll(dto);
  }

  @Get('user/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'ticket'])
  findByUser(@Param('id') id: string, @Query() dto: PaginationAllDto) {
    return this.ticketsService.find(id, dto);
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(...Object.values(UserRole))
  find(@CurrentUser() user: Account, @Query() dto: PaginationAllDto) {
    return this.ticketsService.find(user.id, dto);
  }

  @Get(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(...Object.values(UserRole))
  findOne(@Param('id') id: string) {
    return this.ticketsService.findOne(id);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'ticket'])
  update(
    @Param('id') id: string,
    @Body() dto: TicketAnswerDto,
    @CurrentUser() user: Account,
  ) {
    dto.updatedBy = user.id;
    return this.ticketsService.update(id, dto);
  }
}
