import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';
import { OrderStatus } from 'src/enum';
import { Brackets, Repository } from 'typeorm';
import { SubCategoryDto } from './dto/sub-category.dto';
import { SubCategory } from './entities/sub-category.entity';

@Injectable()
export class SubCategoryService {
  constructor(
    @InjectRepository(SubCategory)
    private readonly repo: Repository<SubCategory>,
  ) {}

  async create(dto: SubCategoryDto) {
    const category = await this.repo.findOne({
      where: { name: dto.name, categoryId: dto.categoryId },
    });
    if (category) {
      throw new ConflictException('Category already exists!');
    }
    const obj = Object.create(dto);
    return this.repo.save(obj);
  }

  async findAll(
    limit: number,
    offset: number,
    keyword: string,
    status: boolean,
    categoryId: string,
  ) {
    const [result, total] = await this.repo
      .createQueryBuilder('subCategory')
      .where(
        'subCategory.status = :status AND subCategory.categoryId = :categoryId',
        { status: status, categoryId: categoryId },
      )
      .andWhere(
        new Brackets((qb) => {
          qb.where('subCategory.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN subCategory.name LIKE '${keyword}%' THEN 0 ELSE 1 END, subCategory.name`,
        'ASC',
      )
      .take(limit)
      .skip(offset)
      .getManyAndCount();

    return { result, total };
  }

  async findSub(limit: number, offset: number, keyword: string) {
    const [result, total] = await this.repo
      .createQueryBuilder('subCategory')
      .where('subCategory.status = :status', { status: true })
      .andWhere(
        new Brackets((qb) => {
          qb.where('subCategory.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN subCategory.name LIKE '${keyword}%' THEN 0 ELSE 1 END, subCategory.name`,
        'ASC',
      )
      .take(limit)
      .skip(offset)
      .getManyAndCount();
    return { result, total };
  }

  async find(
    limit: number,
    offset: number,
    keyword: string,
    categoryId: string,
  ) {
    const [result, total] = await this.repo
      .createQueryBuilder('subCategory')
      .where(
        'subCategory.status = :status AND subCategory.categoryId = :categoryId',
        { status: true, categoryId: categoryId },
      )
      .andWhere(
        new Brackets((qb) => {
          qb.where('subCategory.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN subCategory.name LIKE '${keyword}%' THEN 0 ELSE 1 END, subCategory.name`,
        'ASC',
      )
      .take(limit)
      .skip(offset)
      .getManyAndCount();
    return { result, total };
  }

  async findsubcate(categoryId: string) {
    const result = await this.repo
      .createQueryBuilder('subCategory')
      .where(
        'subCategory.status = :status AND subCategory.categoryId = :categoryId',
        { status: true, categoryId: categoryId },
      )
      .getMany();
    return { result };
  }

  async findByMultiCat(
    limit: number,
    offset: number,
    keyword: string,
    categories: any,
  ) {
    const [result, total] = await this.repo
      .createQueryBuilder('subCategory')
      .where(
        'subCategory.status = :status AND subCategory.categoryId IN :categoryId',
        { status: true, categoryId: categories },
      )
      .andWhere(
        new Brackets((qb) => {
          qb.where('subCategory.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy(
        `CASE WHEN subCategory.name LIKE '${keyword}%' THEN 0 ELSE 1 END, subCategory.name`,
        'ASC',
      )
      .take(limit)
      .skip(offset)
      .getManyAndCount();
    return { result, total };
  }

  async products(
    limit: number,
    offset: number,
    keyword: string,
    status: OrderStatus,
    cartId: string,
    outletDetailId: string,
    outletBranchId: string,
  ) {
    const query = this.repo
      .createQueryBuilder('subCategory')
      .leftJoinAndSelect('subCategory.outletSubCategory', 'outletSubCategory');
    if (cartId && cartId.length > 3 && cartId != 'null') {
      query.leftJoinAndSelect(
        'subCategory.order',
        'order',
        'order.outletDetailId = :outletDetailId AND order.status IN (:...ostatus) AND order.cartId = :cartId',
        {
          outletDetailId: outletDetailId,
          ostatus: [OrderStatus.CART, OrderStatus.ORDERED],
          cartId: cartId,
        },
      );
    } else {
      query.leftJoinAndSelect(
        'subCategory.order',
        'order',
        'order.outletDetailId = :outletDetailId AND order.status IN (:...ostatus) AND order.cartId IS NULL',
        {
          outletDetailId: outletDetailId,
          ostatus: [OrderStatus.CART, OrderStatus.ORDERED],
        },
      );
    }
    const data = await query
      .leftJoinAndSelect('subCategory.category', 'category')
      .select([
        'subCategory.id',
        'subCategory.name',
        'subCategory.image',
        'subCategory.foodType',

        'category.id',
        'category.name',

        'order.id',
        'order.quantity',
        'order.status',
        'order.createdAt',
      ])
      .where(
        'subCategory.status = :status AND outletSubCategory.outletBranchId = :outletBranchId',
        { status: true, outletBranchId: outletBranchId },
      )
      .andWhere(
        new Brackets((qb) => {
          qb.where('subCategory.name LIKE :pname', {
            pname: '%' + keyword + '%',
          });
        }),
      )
      .orderBy({ 'subCategory.name': 'ASC' })
      .take(limit)
      .skip(offset)
      .getRawMany();
    const queryOrder = this.repo
      .createQueryBuilder('subCategory')
      .leftJoinAndSelect('subCategory.outletSubCategory', 'outletSubCategory');
    if (cartId && cartId.length > 3 && cartId != 'null') {
      queryOrder.leftJoinAndSelect(
        'subCategory.order',
        'order',
        'order.outletDetailId = :outletDetailId AND order.status IN (:...ostatus) AND order.cartId = :cartId',
        {
          outletDetailId: outletDetailId,
          ostatus: [OrderStatus.CART, OrderStatus.ORDERED],
          cartId: cartId,
        },
      );
    } else {
      queryOrder.leftJoinAndSelect(
        'subCategory.order',
        'order',
        'order.outletDetailId = :outletDetailId AND order.status IN (:...ostatus) AND order.cartId IS NULL',
        {
          outletDetailId: outletDetailId,
          ostatus: [OrderStatus.CART, OrderStatus.ORDERED],
        },
      );
    }
    const order = await queryOrder
      .select([
        'subCategory.id',

        'order.id',
        'order.cartId',
        'order.status',
        'order.createdAt',
      ])
      .where(
        'subCategory.status = :status AND  outletSubCategory.outletBranchId = :outletBranchId',
        { status: true, outletBranchId: outletBranchId },
      )
      .orderBy({ 'subCategory.name': 'ASC' })
      .getOne();
    return { data, order };
  }

  async findOne(id: string) {
    const category = await this.repo.findOne({ where: { id } });
    if (!category) {
      throw new NotFoundException('Category not found!');
    }
    return category;
  }

  async update(id: string, dto: SubCategoryDto) {
    try {
      const category = await this.repo.findOne({ where: { id } });
      if (!category) {
        throw new NotFoundException('Category not found!');
      }
      const obj = Object.assign(category, {
        name: dto.name,
        foodType: dto.foodType,
      });
      return this.repo.save(obj);
    } catch (error) {
      throw new NotAcceptableException(
        'Either catgeory exists or invalid name!',
      );
    }
  }

  async image(id: string, image: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Order not found!');
    }
    const obj = Object.assign(result, {
      image: process.env.ST_CDN_LINK + image,
      imageName: image,
    });
    return this.repo.save(obj);
  }

  async status(id: string, dto: BoolStatusDto) {
    const menu = await this.repo.findOne({ where: { id } });
    if (!menu) {
      throw new NotFoundException('Category not found!');
    }
    const obj = Object.assign(menu, dto);
    return this.repo.save(obj);
  }
}
