import {
  Body,
  Controller,
  FileTypeValidator,
  Get,
  MaxFileSizeValidator,
  Param,
  ParseFilePipe,
  Patch,
  Post,
  Put,
  Query,
  UploadedFile,
  UseGuards,
  UseInterceptors,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { FileInterceptor } from '@nestjs/platform-express';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';
import { PaginationDto } from 'src/common/dto/pagination.dto';
import { PermissionAction, UserRole } from 'src/enum';
import {
  PaginationPDto,
  PaginationSDto,
  SubCategoryDto,
  ProductPaginationDto,
} from './dto/sub-category.dto';
import { SubCategoryService } from './sub-category.service';

@Controller('sub-category')
export class SubCategoryController {
  constructor(private readonly subCategoryService: SubCategoryService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.CREATE, 'sub_category'])
  create(@Body() dto: SubCategoryDto) {
    return this.subCategoryService.create(dto);
  }

  @Get('list/all/:categoryId')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'sub_category'])
  findAll(
    @Query() query: PaginationSDto,
    @Param('categoryId') categoryId: string,
  ) {
    const keyword = query.keyword || '';
    return this.subCategoryService.findAll(
      query.limit,
      query.offset,
      keyword,
      query.status,
      categoryId,
    );
  }

  @Get('list/:categoryId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  find(@Query() query: PaginationDto, @Param('categoryId') categoryId: string) {
    const keyword = query.keyword || '';
    return this.subCategoryService.find(
      query.limit,
      query.offset,
      keyword,
      categoryId,
    );
  }

  @Get('allsub/:categoryId')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  findsubcate(@Param('categoryId') categoryId: string) {
    return this.subCategoryService.findsubcate(categoryId);
  }

  @Get('list')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  findByMultiCat(@Query() query: PaginationPDto) {
    const keyword = query.keyword || '';
    return this.subCategoryService.findSub(
      query.limit,
      query.offset,
      keyword,
    );
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  products(@Query() query: ProductPaginationDto, @CurrentUser() user: Account) {
    const keyword = query.keyword || '';
    return this.subCategoryService.products(
      query.limit,
      query.offset,
      keyword,
      query.status,
      query.cartId,
      user.outletDetail[0].id,
      query.outletBranchId,
    );
  }

  @Get(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  findOne(@Param('id') id: string) {
    return this.subCategoryService.findOne(id);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'sub_category'])
  update(@Param('id') id: string, @Body() dto: SubCategoryDto) {
    return this.subCategoryService.update(id, dto);
  }

  @Put('status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'sub_category'])
  status(@Param('id') id: string, @Body() dto: BoolStatusDto) {
    return this.subCategoryService.status(id, dto);
  }

  @Put(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @UseInterceptors(FileInterceptor('file'))
  async image2(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new FileTypeValidator({ fileType: '.(png|jpeg|jpg)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    return this.subCategoryService.image(id, file.path);
  }
}
