import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import {
  OutletBranchDto,
  PaginationDto,
  PaginationSDto,
  StatusDto,
} from './dto/outlet-branch.dto';
import { OutletBranch } from './entities/outlet-branch.entity';

@Injectable()
export class OutletBranchService {
  constructor(
    @InjectRepository(OutletBranch)
    private readonly repo: Repository<OutletBranch>,
  ) {}

  async create(dto: OutletBranchDto) {
    const result = await this.repo.findOne({
      where: {
        branchName: dto.branchName,
        companyDetailId: dto.companyDetailId,
      },
    });
    if (result) {
      throw new ConflictException('Outlet already exists!');
    }
    const obj = Object.create(dto);
    return this.repo.save(obj);
  }

  async findAll(dto: PaginationSDto) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('outletBranch')
      .leftJoinAndSelect('outletBranch.companyDetail', 'companyDetail')
      .leftJoinAndSelect('outletBranch.city', 'city')
      .leftJoinAndSelect('outletBranch.state', 'state')
      .where('outletBranch.status = :status', {
        status: dto.status,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where('outletBranch.branchName LIKE :branchName', {
            branchName: '%' + keyword + '%',
          });
        }),
      )
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'outletBranch.branchName': 'ASC' })
      .getManyAndCount();
    return { result, total };
  }

  async find(dto: PaginationSDto, companyId: string) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('outletBranch')
      .leftJoinAndSelect('outletBranch.companyDetail', 'companyDetail')
      .leftJoinAndSelect('outletBranch.city', 'city')
      .leftJoinAndSelect('outletBranch.state', 'state')
      .where('outletBranch.companyDetailId = :companyDetailId AND outletBranch.status = :status', {
        companyDetailId: companyId,
        status: dto.status,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where('outletBranch.branchName LIKE :branchName', {
            branchName: '%' + keyword + '%',
          });
        }),
      )
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'outletBranch.branchName': 'ASC' })
      .getManyAndCount();
    return { result, total };
  }

  async findOne(id: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Outlet not found!');
    }
    return result;
  }

  async update(id: string, dto: OutletBranchDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Outlet not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async status(id: string, dto: StatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Outlet not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }
}
