import {
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Brackets, Repository } from 'typeorm';
import { CreateContactUsDto, UpdateContactUsDto } from './dto/contact-us.dto';
import { PaginationDto } from './dto/pagination.dto';
import { ContactUs } from './entities/contact-us.entity';

@Injectable()
export class ContactUsService {
  constructor(
    @InjectRepository(ContactUs)
    private readonly repo: Repository<ContactUs>,
  ) {}

  async create(dto: CreateContactUsDto) {
    try {
      const obj = Object.create(dto);
      return this.repo.save(obj);
    } catch (error) {
      throw new NotAcceptableException(
        'Something bad happened. Please try after some time!',
      );
    }
  }

  async findAll(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('customerQna')
      .select([
        'customerQna.qnaId',
        'customerQna.answer',
        'customerQna.email',
        'customerQna.body',
        'customerQna.phone',
        'customerQna.status',
        'customerQna.subject',
        'customerQna.createdAt',
        'customerQna.updatedAt',
      ])
      .where('ContactUs.status = :status', { status: dto.status })
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'contactUs.email LIKE :email OR contactUs.subject LIKE :subject OR contactUs.phone LIKE :phone OR contactUs.body LIKE :body OR contactUs.answer LIKE :answer',
            {
              email: '%' + keyword + '%',
              subject: '%' + keyword + '%',
              phone: '%' + keyword + '%',
              body: '%' + keyword + '%',
              answer: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'contactUs.createdAt': 'DESC' })
      .offset(dto.offset)
      .skip(dto.limit)
      .getManyAndCount();
    return { result, total };
  }

  async findOne(contactUsId: string) {
    const result = await this.repo.findOne({ where: { contactUsId } });
    if (!result) {
      throw new NotFoundException('Item not found!');
    }
    return result;
  }

  async update(
    contactUsId: string,
    dto: UpdateContactUsDto,
    accountId: string,
  ) {
    const result = await this.repo.findOne({ where: { contactUsId } });
    if (result) {
      throw new NotFoundException('Item not Found!');
    }
    try {
      const obj = Object.assign(result, { answer: dto.answer, accountId });
      return this.repo.save(obj);
    } catch (error) {
      throw new NotAcceptableException(
        'Something bad happened. Please try after some time!',
      );
    }
  }
}
