import {
  ConflictException,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { CompanyStatus } from 'src/enum';
import { Brackets, Repository } from 'typeorm';
import {
  BusinessDetailDto,
  CompanyDetailDto,
  PaginationDto,
  PaginationSDto,
  StatusDto,
} from './dto/company-detail.dto';
import { CompanyDetail } from './entities/company-detail.entity';

@Injectable()
export class CompanyDetailsService {
  constructor(
    @InjectRepository(CompanyDetail)
    private readonly repo: Repository<CompanyDetail>,
  ) {}

  async create(dto: BusinessDetailDto) {
    const result = await this.repo.findOne({
      where: { businessName: dto.businessName },
    });
    if (result) {
      throw new ConflictException('Company with same name already exists!');
    }
    const obj = Object.create(dto);
    return this.repo.save(obj);
  }

  async findAll(dto: PaginationSDto) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('companyDetail')
      .leftJoinAndSelect('companyDetail.state', 'state')
      .leftJoinAndSelect('companyDetail.city', 'city')
      .where('companyDetail.status = :status', {
        status: dto.status,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'companyDetail.businessName LIKE :businessName OR companyDetail.email LIKE :email',
            {
              businessName: '%' + keyword + '%',
              email: '%' + keyword + '%',
            },
          );
        }),
      )
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'companyDetail.businessName': 'ASC' })
      .getManyAndCount();
    return { result, total };
  }

  async findList(dto: PaginationDto) {
    const category = JSON.parse(dto.category);
    const subcategory = JSON.parse(dto.subCategory);
    const keyword = dto.keyword || '';
    const query = this.repo
      .createQueryBuilder('companyDetail')
      .leftJoinAndSelect('companyDetail.state', 'state')
      .leftJoinAndSelect('companyDetail.city', 'city')
      .leftJoinAndSelect('companyDetail.companyCategory', 'companyCategory')
      .leftJoinAndSelect(
        'companyDetail.companySubCategory',
        'companySubCategory',
      )
      .leftJoinAndSelect('companySubCategory.subCategory', 'subCategory')
      .leftJoinAndSelect('companyCategory.category', 'category')
      .select([
        'companyDetail.id',
        'companyDetail.businessName',
        'companyDetail.personName',
        'companyDetail.logo',

        'state.id',
        'state.name',

        'city.id',
        'city.name',

        'companyCategory.id',
        'category.id',
        'category.name',

        'companySubCategory.id',
        'subCategory.id',
        'subCategory.name',
      ])
      .where('companyDetail.status = :status', {
        status: CompanyStatus.APPROVED,
      });
    if (category.length > 0) {
      query.andWhere('category.id IN :categorys', { categorys: category });
    }
    if (subcategory.length > 0) {
      query.andWhere('subCategory.id IN :subCategorys', {
        subCategorys: subcategory,
      });
    }
    query.andWhere(
      new Brackets((qb) => {
        qb.where('companyDetail.businessName LIKE :businessName', {
          businessName: '%' + keyword + '%',
        });
      }),
    );
    const [result, total] = await query
      .skip(dto.offset)
      .take(dto.limit)
      .orderBy({ 'companyDetail.businessName': 'ASC' })
      .getManyAndCount();
    return { result, total };
  }

  async findCompany(id: string) {
    const result = await this.repo
      .createQueryBuilder('companyDetail')
      .where('companyDetail.accountId = :accountId', { accountId: id })
      .getOne();
    if (!result) {
      throw new NotFoundException('Company not found!');
    }
    return result;
  }

  async findOne(id: string) {
    const result = await this.repo
      .createQueryBuilder('companyDetail')
      .leftJoinAndSelect('companyDetail.companyCategory', 'companyCategory')
      .leftJoinAndSelect('companyDetail.account', 'account')
      .leftJoinAndSelect('companyDetail.state', 'state')
      .leftJoinAndSelect('companyDetail.city', 'city')
      .leftJoinAndSelect('companyCategory.category', 'category')
      .leftJoinAndSelect(
        'companyDetail.companySubCategory',
        'companySubCategory',
      )
      .leftJoinAndSelect('companySubCategory.subCategory', 'subCategory')
      .where('companyDetail.id = :id', { id: id })
      .getOne();
    if (!result) {
      throw new NotFoundException('Company not found!');
    }
    return result;
  }

  async info(id: string) {
    const result = await this.repo
      .createQueryBuilder('companyDetail')
      .leftJoinAndSelect('companyDetail.state', 'state')
      .leftJoinAndSelect('companyDetail.city', 'city')
      .leftJoinAndSelect('companyDetail.companyCategory', 'companyCategory')
      .leftJoinAndSelect(
        'companyDetail.companySubCategory',
        'companySubCategory',
      )
      .leftJoinAndSelect('companySubCategory', 'subCategory')
      .leftJoinAndSelect('companyCategory', 'category')
      .select([
        'companyDetail.id',
        'companyDetail.businessName',
        'companyDetail.personName',
        'companyDetail.logo',
        'companyDetail.desc',

        'state.id',
        'state.name',

        'city.id',
        'city.name',

        'companyCategory.id',
        'category.id',
        'category.name',

        'companySubCategory.id',
        'subCategory.id',
        'subCategory.name',
      ])
      .where(
        'companyDetail.status = :status AND companyDetail.accountId = :accountId',
        {
          status: CompanyStatus.APPROVED,
          accountId: id,
        },
      )
      .getOne();
    if (!result) {
      throw new NotFoundException('Company not found!');
    }
    return result;
  }

  async update(id: string, dto: CompanyDetailDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Company not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async file(id: string, image: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Company detail not found!');
    }
    const obj = Object.assign(result, {
      file: process.env.CDN_LINK + image,
      filePath: image,
    });
    return this.repo.save(obj);
  }

  async logo(id: string, image: string) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Company detail not found!');
    }
    const obj = Object.assign(result, {
      logo: process.env.CDN_LINK + image,
      logoPath: image,
    });
    return this.repo.save(obj);
  }

  async status(id: string, dto: StatusDto) {
    const result = await this.repo.findOne({ where: { id } });
    if (!result) {
      throw new NotFoundException('Company detail not found!');
    }
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }
}
