import {
  Body,
  Controller,
  Get,
  Param,
  Patch,
  Post,
  Put,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { PaginationDto } from 'src/account/dto/account.dto';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import { CityService } from './city.service';
import { CityDto, PaginationSDto } from './dto/city.dto';
import { BoolStatusDto } from 'src/common/dto/bool-status.dto';

@Controller('city')
export class CityController {
  constructor(private readonly cityService: CityService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.CREATE, 'city'])
  create(@Body() dto: CityDto) {
    return this.cityService.create(dto);
  }

  @Get('list/all/:stateId')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'city'])
  findAll(@Query() query: PaginationSDto, @Param('stateId') stateId: string) {
    const keyword = query.keyword || '';
    return this.cityService.findAll(
      query.limit,
      query.offset,
      keyword,
      query.status,
      +stateId,
    );
  }

  @Get('list/:stateId')
  find(@Query() query: PaginationDto, @Param('stateId') stateId: string) {
    const keyword = query.keyword || '';
    return this.cityService.find(query.limit, query.offset, keyword, +stateId);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'city'])
  update(@Param('id') id: string, @Body() dto: CityDto) {
    return this.cityService.update(+id, dto);
  }

  @Put('status/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'city'])
  status(@Param('id') id: string, @Body() dto: BoolStatusDto) {
    return this.cityService.status(+id, dto);
  }
}
