import {
  ConflictException,
  Injectable,
  NotAcceptableException,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import * as bcrypt from 'bcrypt';
import { DefaultStatus, UserRole } from 'src/enum';
import { StaffDetail } from 'src/staff-details/entities/staff-detail.entity';
import { Brackets, Repository } from 'typeorm';
import { CreateAccountDto, PaginationDto, StatusDto, PasswordDto, PasswordWithOldDto } from './dto/account.dto';
import { Account } from './entities/account.entity';

@Injectable()
export class AccountService {
  constructor(
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(StaffDetail)
    private readonly staffRepo: Repository<StaffDetail>,
  ) {}

  async create(dto: CreateAccountDto, createdBy: string) {
    const user = await this.repo.findOne({
      where: { loginId: dto.loginId, roles: dto.roles },
    });
    if (user) {
      throw new ConflictException('Login id already exists!');
    }
    const encryptedPassword = await bcrypt.hash(dto.password, 13);
    const obj = Object.create({
      loginId: dto.loginId,
      password: encryptedPassword,
      createdBy,
      roles: dto.roles,
      employeeId: dto.employeeId,
      passwordShow: dto.password,
    });
    const payload = await this.repo.save(obj);
    const object = Object.create({
      name: dto.name,
      emailId:dto.emailId,
      gender: dto.gender,
      dob: dto.dob,
      accountId: payload.id,
      phone: dto.phone,
    });
    if (
      dto.roles === UserRole.STAFF ||
      dto.roles === UserRole.ADMIN ||
      dto.roles === UserRole.SUPER_ADMIN ||
      dto.roles === UserRole.DELIVERY_BOY
    ) {
      this.staffRepo.save(object);
    }
    return payload;
  }

  async outletDetail(loginId: string, password: string, createdBy: string) {
    const user = await this.repo.findOne({
      where: { loginId: loginId, roles: UserRole.SUB_OUTLET },
    });
    if (user) {
      throw new ConflictException('Login id already exists!');
    }
    const encryptedPassword = await bcrypt.hash(password, 13);
    const obj = Object.create({
      loginId: loginId,
      password: encryptedPassword,
      passwordShow: password,
      createdBy,
      roles: UserRole.SUB_OUTLET,
    });
    return this.repo.save(obj);
  }

  async find(dto: PaginationDto, createdBy: string) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .where(
        'account.status = :status AND account.roles = :roles AND account.createdBy = :createdBy',
        {
          status: dto.status,
          roles: dto.role,
          createdBy: createdBy,
        },
      )
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.loginId LIKE :loginId OR staffDetail.name LIKE :pname',
            {
              loginId: '%' + keyword + '%',
              pname: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'staffDetail.name': 'ASC' })
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();
    return { result, total };
  }

  async findAll(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.companyDetail', 'companyDetail')
      .where('account.status = :status AND account.roles = :roles', {
        status: dto.status,
        roles: dto.role,
      })
      .andWhere(
        new Brackets((qb) => {
          qb.where(
            'account.phoneNumber LIKE :phoneNumber OR companyDetail.name LIKE :dname',
            {
              phoneNumber: '%' + keyword + '%',
              dname: '%' + keyword + '%',
            },
          );
        }),
      )
      .orderBy({ 'companyDetail.name': 'ASC' })
      .skip(dto.offset)
      .take(dto.limit)
      .getManyAndCount();

    return { result, total };
  }

  async outletProfile(id: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.outletDetail', 'outletDetail')
      .leftJoinAndSelect('outletDetail.city', 'city')
      .leftJoinAndSelect('outletDetail.state', 'state')
      .leftJoinAndSelect('outletDetail.area', 'area')
      .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
      .leftJoinAndSelect('outletBranch.city', 'ocity')
      .leftJoinAndSelect('outletBranch.state', 'ostate')
      .leftJoinAndSelect('outletBranch.companyDetail', 'companyDetail')
      .select([
        'account.id',

        'outletDetail.id',
        'outletDetail.outletId',
        'outletDetail.storeName',
        'outletDetail.contactPerson',
        'outletDetail.address',
        'outletDetail.latitude',
        'outletDetail.longitude',
        'outletDetail.storeOpenTime',
        'outletDetail.orderOpenTime',
        'outletDetail.orderCloseTime',
        'outletDetail.type',
        'outletDetail.status',

        'outletBranch.id',
        'outletBranch.branchName',
        'outletBranch.contactPersonName',
        'outletBranch.email',
        'outletBranch.phone',
        'outletBranch.logo',
        'outletBranch.status',

        'companyDetail.id',
        'companyDetail.businessName',
        'companyDetail.logo',
        'companyDetail.email',
        'companyDetail.desc',
        'companyDetail.address',
        'companyDetail.status',

        'area.id',
        'area.name',
        'city.id',
        'city.name',
        'state.id',
        'state.name',
      ])
      .where('account.id = :id AND account.roles = :roles', {
        id: id,
        roles: UserRole.SUB_OUTLET,
      })
      .getOne();

    if (!result) {
      throw new NotFoundException('Profile not found!');
    }
    return result;
  }

  async deliveryProfile(id: string) {
    const result = await this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.staffDetail', 'staffDetail')
      .select([
        'account.id',
        'account.employeeId',
        'staffDetail.id',
        'staffDetail.name',
        'staffDetail.phone',
        'staffDetail.designation',
        'staffDetail.panNo',
        'staffDetail.aadharNo',
        'staffDetail.profile',
        'staffDetail.pan',
        'staffDetail.aadhar',
        'staffDetail.gender',
        'staffDetail.dob',
      ])
      .where('account.id = :id AND account.roles = :roles', {
        id: id,
        roles: UserRole.DELIVERY_BOY,
      })
      .getOne();

    if (!result) {
      throw new NotFoundException('Profile not found!');
    }
    return result;
  }

  async findOne(id: string) {
    const user = await this.repo.findOne({
      relations: ['staffDetail'],
      where: { id },
    });
    if (!user) {
      throw new ConflictException('User not found!');
    }

    return user;
  }

  async updateOwnPassword(dto: PasswordWithOldDto, id: string) {
    const user = await this.repo.findOne({ where: { id } });

    if (!user) {
      throw new ConflictException('User detail not found!');
    }

    const comparePassword = await bcrypt.compare(
      dto.oldPassword,
      user.password,
    );
    if (!comparePassword) {
      throw new NotAcceptableException('Incorrect old password!');
    }

    const epassword = await bcrypt.hash(dto.password, 13);

    const obj = Object.assign(user, {
      password: epassword,
      passwordShow: dto.password,
    });
    return this.repo.save(obj);
  }

  async updatePassword(dto: PasswordDto, id: string) {
    const user = await this.repo.findOne({ where: { id } });

    if (!user) {
      throw new ConflictException('User detail not found!');
    }
    const epassword = await bcrypt.hash(dto.password, 13);

    const obj = Object.assign(user, {
      password: epassword,
      passwordShow: dto.password,
    });
    return this.repo.save(obj);
  }

  async status(id: string, status: StatusDto) {
    const user = await this.repo.findOne({ where: { id } });
    if (!user) {
      throw new NotFoundException('User not found!');
    }
    const obj = Object.assign(user, status);
    return this.repo.save(obj);
  }

  async remove(id: string) {
    const user = await this.repo.findOne({ where: { id } });
    if (!user) {
      throw new NotFoundException('User not found!');
    }
    const obj = Object.assign(user, { status: DefaultStatus.DELETED });
    return this.repo.save(user);
  }
}
