"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpdateRecorderBom = exports.UpdateRecorderBase = void 0;
const exception_1 = require("../exception/exception");
const update_buffer_1 = require("../utility/update-buffer");
class UpdateRecorderBase {
    constructor(entry) {
        this._original = Buffer.from(entry.content);
        this._content = update_buffer_1.UpdateBufferBase.create(entry.path, entry.content);
        this._path = entry.path;
    }
    static createFromFileEntry(entry) {
        const c0 = entry.content.byteLength > 0 && entry.content.readUInt8(0);
        const c1 = entry.content.byteLength > 1 && entry.content.readUInt8(1);
        const c2 = entry.content.byteLength > 2 && entry.content.readUInt8(2);
        // Check if we're BOM.
        if (c0 == 0xef && c1 == 0xbb && c2 == 0xbf) {
            return new UpdateRecorderBom(entry);
        }
        else if (c0 === 0xff && c1 == 0xfe) {
            return new UpdateRecorderBom(entry);
        }
        else if (c0 === 0xfe && c1 == 0xff) {
            return new UpdateRecorderBom(entry);
        }
        return new UpdateRecorderBase(entry);
    }
    get path() {
        return this._path;
    }
    // These just record changes.
    insertLeft(index, content) {
        this._content.insertLeft(index, typeof content == 'string' ? Buffer.from(content) : content);
        return this;
    }
    insertRight(index, content) {
        this._content.insertRight(index, typeof content == 'string' ? Buffer.from(content) : content);
        return this;
    }
    remove(index, length) {
        this._content.remove(index, length);
        return this;
    }
    apply(content) {
        if (!content.equals(this._content.original)) {
            throw new exception_1.ContentHasMutatedException(this.path);
        }
        return this._content.generate();
    }
}
exports.UpdateRecorderBase = UpdateRecorderBase;
class UpdateRecorderBom extends UpdateRecorderBase {
    constructor(entry, _delta = 1) {
        super(entry);
        this._delta = _delta;
    }
    insertLeft(index, content) {
        return super.insertLeft(index + this._delta, content);
    }
    insertRight(index, content) {
        return super.insertRight(index + this._delta, content);
    }
    remove(index, length) {
        return super.remove(index + this._delta, length);
    }
}
exports.UpdateRecorderBom = UpdateRecorderBom;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVjb3JkZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9wYWNrYWdlcy9hbmd1bGFyX2RldmtpdC9zY2hlbWF0aWNzL3NyYy90cmVlL3JlY29yZGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7O0dBTUc7OztBQUVILHNEQUFvRTtBQUNwRSw0REFBNEQ7QUFHNUQsTUFBYSxrQkFBa0I7SUFLN0IsWUFBWSxLQUFnQjtRQUMxQixJQUFJLENBQUMsU0FBUyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzVDLElBQUksQ0FBQyxRQUFRLEdBQUcsZ0NBQWdCLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ25FLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztJQUMxQixDQUFDO0lBRUQsTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCO1FBQ3pDLE1BQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxHQUFHLENBQUMsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN0RSxNQUFNLEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLFVBQVUsR0FBRyxDQUFDLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEUsTUFBTSxFQUFFLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxVQUFVLEdBQUcsQ0FBQyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXRFLHNCQUFzQjtRQUN0QixJQUFJLEVBQUUsSUFBSSxJQUFJLElBQUksRUFBRSxJQUFJLElBQUksSUFBSSxFQUFFLElBQUksSUFBSSxFQUFFO1lBQzFDLE9BQU8sSUFBSSxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUNyQzthQUFNLElBQUksRUFBRSxLQUFLLElBQUksSUFBSSxFQUFFLElBQUksSUFBSSxFQUFFO1lBQ3BDLE9BQU8sSUFBSSxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUNyQzthQUFNLElBQUksRUFBRSxLQUFLLElBQUksSUFBSSxFQUFFLElBQUksSUFBSSxFQUFFO1lBQ3BDLE9BQU8sSUFBSSxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUNyQztRQUVELE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQsSUFBSSxJQUFJO1FBQ04sT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDO0lBQ3BCLENBQUM7SUFFRCw2QkFBNkI7SUFDN0IsVUFBVSxDQUFDLEtBQWEsRUFBRSxPQUF3QjtRQUNoRCxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsT0FBTyxPQUFPLElBQUksUUFBUSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUU3RixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRCxXQUFXLENBQUMsS0FBYSxFQUFFLE9BQXdCO1FBQ2pELElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxPQUFPLE9BQU8sSUFBSSxRQUFRLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBRTlGLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVELE1BQU0sQ0FBQyxLQUFhLEVBQUUsTUFBYztRQUNsQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsS0FBSyxDQUFDLE9BQWU7UUFDbkIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUMzQyxNQUFNLElBQUksc0NBQTBCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ2pEO1FBRUQsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDO0lBQ2xDLENBQUM7Q0FDRjtBQTFERCxnREEwREM7QUFFRCxNQUFhLGlCQUFrQixTQUFRLGtCQUFrQjtJQUN2RCxZQUFZLEtBQWdCLEVBQVUsU0FBUyxDQUFDO1FBQzlDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUR1QixXQUFNLEdBQU4sTUFBTSxDQUFJO0lBRWhELENBQUM7SUFFUSxVQUFVLENBQUMsS0FBYSxFQUFFLE9BQXdCO1FBQ3pELE9BQU8sS0FBSyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRVEsV0FBVyxDQUFDLEtBQWEsRUFBRSxPQUF3QjtRQUMxRCxPQUFPLEtBQUssQ0FBQyxXQUFXLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVRLE1BQU0sQ0FBQyxLQUFhLEVBQUUsTUFBYztRQUMzQyxPQUFPLEtBQUssQ0FBQyxNQUFNLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDbkQsQ0FBQztDQUNGO0FBaEJELDhDQWdCQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQgeyBDb250ZW50SGFzTXV0YXRlZEV4Y2VwdGlvbiB9IGZyb20gJy4uL2V4Y2VwdGlvbi9leGNlcHRpb24nO1xuaW1wb3J0IHsgVXBkYXRlQnVmZmVyQmFzZSB9IGZyb20gJy4uL3V0aWxpdHkvdXBkYXRlLWJ1ZmZlcic7XG5pbXBvcnQgeyBGaWxlRW50cnksIFVwZGF0ZVJlY29yZGVyIH0gZnJvbSAnLi9pbnRlcmZhY2UnO1xuXG5leHBvcnQgY2xhc3MgVXBkYXRlUmVjb3JkZXJCYXNlIGltcGxlbWVudHMgVXBkYXRlUmVjb3JkZXIge1xuICBwcm90ZWN0ZWQgX3BhdGg6IHN0cmluZztcbiAgcHJvdGVjdGVkIF9vcmlnaW5hbDogQnVmZmVyO1xuICBwcm90ZWN0ZWQgX2NvbnRlbnQ6IFVwZGF0ZUJ1ZmZlckJhc2U7XG5cbiAgY29uc3RydWN0b3IoZW50cnk6IEZpbGVFbnRyeSkge1xuICAgIHRoaXMuX29yaWdpbmFsID0gQnVmZmVyLmZyb20oZW50cnkuY29udGVudCk7XG4gICAgdGhpcy5fY29udGVudCA9IFVwZGF0ZUJ1ZmZlckJhc2UuY3JlYXRlKGVudHJ5LnBhdGgsIGVudHJ5LmNvbnRlbnQpO1xuICAgIHRoaXMuX3BhdGggPSBlbnRyeS5wYXRoO1xuICB9XG5cbiAgc3RhdGljIGNyZWF0ZUZyb21GaWxlRW50cnkoZW50cnk6IEZpbGVFbnRyeSk6IFVwZGF0ZVJlY29yZGVyQmFzZSB7XG4gICAgY29uc3QgYzAgPSBlbnRyeS5jb250ZW50LmJ5dGVMZW5ndGggPiAwICYmIGVudHJ5LmNvbnRlbnQucmVhZFVJbnQ4KDApO1xuICAgIGNvbnN0IGMxID0gZW50cnkuY29udGVudC5ieXRlTGVuZ3RoID4gMSAmJiBlbnRyeS5jb250ZW50LnJlYWRVSW50OCgxKTtcbiAgICBjb25zdCBjMiA9IGVudHJ5LmNvbnRlbnQuYnl0ZUxlbmd0aCA+IDIgJiYgZW50cnkuY29udGVudC5yZWFkVUludDgoMik7XG5cbiAgICAvLyBDaGVjayBpZiB3ZSdyZSBCT00uXG4gICAgaWYgKGMwID09IDB4ZWYgJiYgYzEgPT0gMHhiYiAmJiBjMiA9PSAweGJmKSB7XG4gICAgICByZXR1cm4gbmV3IFVwZGF0ZVJlY29yZGVyQm9tKGVudHJ5KTtcbiAgICB9IGVsc2UgaWYgKGMwID09PSAweGZmICYmIGMxID09IDB4ZmUpIHtcbiAgICAgIHJldHVybiBuZXcgVXBkYXRlUmVjb3JkZXJCb20oZW50cnkpO1xuICAgIH0gZWxzZSBpZiAoYzAgPT09IDB4ZmUgJiYgYzEgPT0gMHhmZikge1xuICAgICAgcmV0dXJuIG5ldyBVcGRhdGVSZWNvcmRlckJvbShlbnRyeSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBVcGRhdGVSZWNvcmRlckJhc2UoZW50cnkpO1xuICB9XG5cbiAgZ2V0IHBhdGgoKSB7XG4gICAgcmV0dXJuIHRoaXMuX3BhdGg7XG4gIH1cblxuICAvLyBUaGVzZSBqdXN0IHJlY29yZCBjaGFuZ2VzLlxuICBpbnNlcnRMZWZ0KGluZGV4OiBudW1iZXIsIGNvbnRlbnQ6IEJ1ZmZlciB8IHN0cmluZyk6IFVwZGF0ZVJlY29yZGVyIHtcbiAgICB0aGlzLl9jb250ZW50Lmluc2VydExlZnQoaW5kZXgsIHR5cGVvZiBjb250ZW50ID09ICdzdHJpbmcnID8gQnVmZmVyLmZyb20oY29udGVudCkgOiBjb250ZW50KTtcblxuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgaW5zZXJ0UmlnaHQoaW5kZXg6IG51bWJlciwgY29udGVudDogQnVmZmVyIHwgc3RyaW5nKTogVXBkYXRlUmVjb3JkZXIge1xuICAgIHRoaXMuX2NvbnRlbnQuaW5zZXJ0UmlnaHQoaW5kZXgsIHR5cGVvZiBjb250ZW50ID09ICdzdHJpbmcnID8gQnVmZmVyLmZyb20oY29udGVudCkgOiBjb250ZW50KTtcblxuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgcmVtb3ZlKGluZGV4OiBudW1iZXIsIGxlbmd0aDogbnVtYmVyKTogVXBkYXRlUmVjb3JkZXIge1xuICAgIHRoaXMuX2NvbnRlbnQucmVtb3ZlKGluZGV4LCBsZW5ndGgpO1xuXG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBhcHBseShjb250ZW50OiBCdWZmZXIpOiBCdWZmZXIge1xuICAgIGlmICghY29udGVudC5lcXVhbHModGhpcy5fY29udGVudC5vcmlnaW5hbCkpIHtcbiAgICAgIHRocm93IG5ldyBDb250ZW50SGFzTXV0YXRlZEV4Y2VwdGlvbih0aGlzLnBhdGgpO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLl9jb250ZW50LmdlbmVyYXRlKCk7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIFVwZGF0ZVJlY29yZGVyQm9tIGV4dGVuZHMgVXBkYXRlUmVjb3JkZXJCYXNlIHtcbiAgY29uc3RydWN0b3IoZW50cnk6IEZpbGVFbnRyeSwgcHJpdmF0ZSBfZGVsdGEgPSAxKSB7XG4gICAgc3VwZXIoZW50cnkpO1xuICB9XG5cbiAgb3ZlcnJpZGUgaW5zZXJ0TGVmdChpbmRleDogbnVtYmVyLCBjb250ZW50OiBCdWZmZXIgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gc3VwZXIuaW5zZXJ0TGVmdChpbmRleCArIHRoaXMuX2RlbHRhLCBjb250ZW50KTtcbiAgfVxuXG4gIG92ZXJyaWRlIGluc2VydFJpZ2h0KGluZGV4OiBudW1iZXIsIGNvbnRlbnQ6IEJ1ZmZlciB8IHN0cmluZykge1xuICAgIHJldHVybiBzdXBlci5pbnNlcnRSaWdodChpbmRleCArIHRoaXMuX2RlbHRhLCBjb250ZW50KTtcbiAgfVxuXG4gIG92ZXJyaWRlIHJlbW92ZShpbmRleDogbnVtYmVyLCBsZW5ndGg6IG51bWJlcikge1xuICAgIHJldHVybiBzdXBlci5yZW1vdmUoaW5kZXggKyB0aGlzLl9kZWx0YSwgbGVuZ3RoKTtcbiAgfVxufVxuIl19