"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSinkBase = void 0;
const rxjs_1 = require("rxjs");
const exception_1 = require("../exception/exception");
const action_1 = require("../tree/action");
const Noop = function () { };
class SimpleSinkBase {
    constructor() {
        this.preCommitAction = Noop;
        this.postCommitAction = Noop;
        this.preCommit = Noop;
        this.postCommit = Noop;
    }
    _fileAlreadyExistException(path) {
        throw new exception_1.FileAlreadyExistException(path);
    }
    _fileDoesNotExistException(path) {
        throw new exception_1.FileDoesNotExistException(path);
    }
    _validateOverwriteAction(action) {
        return this._validateFileExists(action.path).pipe((0, rxjs_1.map)((b) => {
            if (!b) {
                this._fileDoesNotExistException(action.path);
            }
        }));
    }
    _validateCreateAction(action) {
        return this._validateFileExists(action.path).pipe((0, rxjs_1.map)((b) => {
            if (b) {
                this._fileAlreadyExistException(action.path);
            }
        }));
    }
    _validateRenameAction(action) {
        return this._validateFileExists(action.path).pipe((0, rxjs_1.map)((b) => {
            if (!b) {
                this._fileDoesNotExistException(action.path);
            }
        }), (0, rxjs_1.mergeMap)(() => this._validateFileExists(action.to)), (0, rxjs_1.map)((b) => {
            if (b) {
                this._fileAlreadyExistException(action.to);
            }
        }));
    }
    _validateDeleteAction(action) {
        return this._validateFileExists(action.path).pipe((0, rxjs_1.map)((b) => {
            if (!b) {
                this._fileDoesNotExistException(action.path);
            }
        }));
    }
    validateSingleAction(action) {
        switch (action.kind) {
            case 'o':
                return this._validateOverwriteAction(action);
            case 'c':
                return this._validateCreateAction(action);
            case 'r':
                return this._validateRenameAction(action);
            case 'd':
                return this._validateDeleteAction(action);
            default:
                throw new action_1.UnknownActionException(action);
        }
    }
    commitSingleAction(action) {
        return (0, rxjs_1.concat)(this.validateSingleAction(action), new rxjs_1.Observable((observer) => {
            let committed = null;
            switch (action.kind) {
                case 'o':
                    committed = this._overwriteFile(action.path, action.content);
                    break;
                case 'c':
                    committed = this._createFile(action.path, action.content);
                    break;
                case 'r':
                    committed = this._renameFile(action.path, action.to);
                    break;
                case 'd':
                    committed = this._deleteFile(action.path);
                    break;
            }
            if (committed) {
                committed.subscribe(observer);
            }
            else {
                observer.complete();
            }
        })).pipe((0, rxjs_1.ignoreElements)());
    }
    commit(tree) {
        const actions = (0, rxjs_1.from)(tree.actions);
        return (0, rxjs_1.concat)(this.preCommit() || (0, rxjs_1.of)(null), (0, rxjs_1.defer)(() => actions).pipe((0, rxjs_1.concatMap)((action) => {
            const maybeAction = this.preCommitAction(action);
            if ((0, rxjs_1.isObservable)(maybeAction) || isPromiseLike(maybeAction)) {
                return maybeAction;
            }
            return (0, rxjs_1.of)(maybeAction || action);
        }), (0, rxjs_1.concatMap)((action) => {
            return (0, rxjs_1.concat)(this.commitSingleAction(action).pipe((0, rxjs_1.ignoreElements)()), (0, rxjs_1.of)(action));
        }), (0, rxjs_1.concatMap)((action) => this.postCommitAction(action) || (0, rxjs_1.of)(null))), (0, rxjs_1.defer)(() => this._done()), (0, rxjs_1.defer)(() => this.postCommit() || (0, rxjs_1.of)(null))).pipe((0, rxjs_1.ignoreElements)(), (0, rxjs_1.defaultIfEmpty)(undefined));
    }
}
exports.SimpleSinkBase = SimpleSinkBase;
function isPromiseLike(value) {
    return !!value && typeof value.then === 'function';
}
//# sourceMappingURL=data:application/json;base64,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