"use strict";
/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createVirtualAstObject = void 0;
const json_1 = require("../../json");
function createVirtualAstObject(root, options = {}) {
    const reporter = (path, target, oldValue, newValue) => {
        if (!options.listener) {
            return;
        }
        if (oldValue === newValue || JSON.stringify(oldValue) === JSON.stringify(newValue)) {
            // same value
            return;
        }
        if (Array.isArray(target)) {
            // For arrays we remove the index and update the entire value as keeping
            // track of changes by indices can be rather complex.
            options.listener(path.slice(0, -1), target);
        }
        else {
            options.listener(path, newValue);
        }
    };
    return create(Array.isArray(root) ? [...root] : { ...root }, [], reporter, new Set(options.exclude), options.include?.length ? new Set(options.include) : undefined);
}
exports.createVirtualAstObject = createVirtualAstObject;
function create(obj, path, reporter, excluded = new Set(), included) {
    return new Proxy(obj, {
        getOwnPropertyDescriptor(target, p) {
            if (excluded.has(p) || (included && !included.has(p))) {
                return undefined;
            }
            return Reflect.getOwnPropertyDescriptor(target, p);
        },
        has(target, p) {
            if (typeof p === 'symbol' || excluded.has(p)) {
                return false;
            }
            return Reflect.has(target, p);
        },
        get(target, p) {
            if (excluded.has(p) || (included && !included.has(p))) {
                return undefined;
            }
            const value = Reflect.get(target, p);
            if (typeof p === 'symbol') {
                return value;
            }
            if (((0, json_1.isJsonObject)(value) && !(value instanceof Map)) || Array.isArray(value)) {
                return create(value, [...path, p], reporter);
            }
            else {
                return value;
            }
        },
        set(target, p, value) {
            if (excluded.has(p) || (included && !included.has(p))) {
                return false;
            }
            if (value === undefined) {
                // setting to undefined is equivalent to a delete.
                return this.deleteProperty?.(target, p) ?? false;
            }
            if (typeof p === 'symbol') {
                return Reflect.set(target, p, value);
            }
            const existingValue = getCurrentValue(target, p);
            if (Reflect.set(target, p, value)) {
                reporter([...path, p], target, existingValue, value);
                return true;
            }
            return false;
        },
        deleteProperty(target, p) {
            if (excluded.has(p)) {
                return false;
            }
            if (typeof p === 'symbol') {
                return Reflect.deleteProperty(target, p);
            }
            const existingValue = getCurrentValue(target, p);
            if (Reflect.deleteProperty(target, p)) {
                reporter([...path, p], target, existingValue, undefined);
                return true;
            }
            return true;
        },
        defineProperty(target, p, attributes) {
            if (typeof p === 'symbol') {
                return Reflect.defineProperty(target, p, attributes);
            }
            return false;
        },
        ownKeys(target) {
            return Reflect.ownKeys(target).filter((p) => !excluded.has(p) && (!included || included.has(p)));
        },
    });
}
function getCurrentValue(target, property) {
    if (Array.isArray(target) && isFinite(+property)) {
        return target[+property];
    }
    if (target && property in target) {
        return target[property];
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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