import { CACHE_MANAGER } from '@nestjs/cache-manager';
import {
  ConflictException,
  Inject,
  Injectable,
  NotFoundException,
} from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Cache } from 'cache-manager';
import { Like, Repository } from 'typeorm';
import { PaginationDto } from './dto/pagination.dto';
import { SettingDto } from './dto/setting.dto';
import { StatusSettingDto } from './dto/status-setting.dto';
import { Setting } from './entities/setting.entity';

@Injectable()
export class SettingsService {
  constructor(
    @InjectRepository(Setting) private readonly repo: Repository<Setting>,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
  ) {}

  async findAll(dto: PaginationDto) {
    const keyword = dto.keyword || '';
    const [result, total] = await this.repo.findAndCount({
      where: [{ title: Like('%' + keyword + '%'), status: dto.status }],
      skip: dto.offset,
      take: dto.limit,
      order: { title: 'ASC' },
    });
    return { result, total };
  }

  async findOne(id: string) {
    return this.getSetting(id);
  }

  async findSetting(domain: string) {
    return this.getSetting(domain);
  }

  async update(id: string, dto: SettingDto) {
    const result = await this.getSetting(id);

    this.deleteSetting(id);
    const obj = Object.assign(result, dto);
    return this.repo.save(obj);
  }

  async status(id: string, dto: StatusSettingDto) {
    const result = await this.getSetting(id);
    const obj = Object.assign(result, dto);
    this.deleteSetting(id);
    return this.repo.save(obj);
  }

  private async deleteSetting(id: string) {
    this.cacheManager.del('setting' + id);
  }

  async getAppSetting(id: string) {
    const result = await this.repo
        .createQueryBuilder('setting')
        .select([
          'setting.id',
          'setting.title',
          'setting.version',
          'setting.logo',
          'setting.logoName',
          'setting.status',
          'setting.maintenance',
          'setting.createdAt',
          'setting.updatedAt',
        ])
        .where('setting.id = :id', { id })
        .getOne();
        
    if (!result) {
      throw new NotFoundException(
        `Something bad happened! Please contact to admin!`,
      );
    }
    return { result };
  }

  private async getSetting(id: string) {
    let result: Setting = await this.cacheManager.get('setting' + id);
    if (!result) {
      result = await this.repo
        .createQueryBuilder('setting')
        .select([
          'setting.id',
          'setting.title',
          'setting.version',
          'setting.logo',
          'setting.logoName',
          'setting.status',
          'setting.createdAt',
          'setting.updatedAt',
        ])
        .where('setting.id = :id', { id })
        .getOne();
      this.cacheManager.set('setting' + id, result, 7 * 24 * 60 * 60 * 1000);
    }
    if (!result) {
      throw new NotFoundException(
        `Something bad happened! Please contact to admin!`,
      );
    }
    return result;
  }
}
