import {
  Body,
  Controller,
  Get,
  Param,
  Patch,
  Post,
  Put,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PermissionAction, UserRole } from 'src/enum';
import {
  PaginationSDto,
  StatusDto,
} from 'src/outlet-branch/dto/outlet-branch.dto';
import {
  CreateOutletDetailDto,
  OutletDetailDto,
  TimeIncreaseDto,
} from './dto/outlet-detail.dto';
import { OutletDetailsService } from './outlet-details.service';

@Controller('outlet-details')
export class OutletDetailsController {
  constructor(private readonly outletDetailsService: OutletDetailsService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.CREATE, 'outlet_detail'])
  create(
    @Body() createOutletDetailDto: CreateOutletDetailDto,
    @CurrentUser() user: Account,
  ) {
    return this.outletDetailsService.create(createOutletDetailDto, user.id);
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'outlet_detail'])
  findAll(@Query() dto: PaginationSDto) {
    return this.outletDetailsService.findAll(dto);
  }

  @Get(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.READ, 'outlet_detail'])
  outletProfile(@Param('id') id: string) {
    return this.outletDetailsService.outletProfile(id);
  }

  @Get('list/:branchId')
  find(@Query() dto: PaginationSDto, @Param('branchId') branchId: string) {
    return this.outletDetailsService.find(dto, branchId);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'outlet_detail'])
  update(
    @Param('id') id: string,
    @Body() updateOutletDetailDto: OutletDetailDto,
  ) {
    return this.outletDetailsService.update(id, updateOutletDetailDto);
  }

  @Put(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'outlet_detail'])
  status(@Param('id') id: string, @Body() dto: StatusDto) {
    return this.outletDetailsService.status(id, dto);
  }

  @Put('time-increase/:id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(...Object.values(UserRole))
  @CheckPermissions([PermissionAction.UPDATE, 'outlet_detail'])
  timeIncrease(@Param('id') id: string, @Body() dto: TimeIncreaseDto) {
    return this.outletDetailsService.timeIncrease(id, dto);
  }
}
