import {
  Body,
  Controller,
  Get,
  Param,
  Patch,
  Post,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
import { PaginationDto } from 'src/common/dto/pagination.dto';
import { PermissionAction, UserRole } from 'src/enum';
import { FaqAnswerDto, FaqDto, PaginationAllDto } from './dto/faq.dto';
import { FaqsService } from './faqs.service';

@Controller('faqs')
export class FaqsController {
  constructor(private readonly faqsService: FaqsService) {}

  @Post()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  create(@Body() dto: FaqDto, @CurrentUser() user: Account) {
    dto.accountId = user.id;
    dto.settingId = user.settingId;
    return this.faqsService.create(dto);
  }

  @Get('all')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'faq'])
  findAll(@Query() dto: PaginationAllDto) {
    return this.faqsService.findAll(dto);
  }

  @Get()
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(...Object.values(UserRole))
  find(@CurrentUser() user: Account, @Query() dto: PaginationDto) {
    return this.faqsService.find(user.settingId, dto);
  }

  @Patch(':id')
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'faq'])
  update(
    @Param('id') id: string,
    @Body() dto: FaqAnswerDto,
    @CurrentUser() user: Account,
  ) {
    dto.updatedBy = user.id;
    return this.faqsService.update(id, dto);
  }
}
