import {
  Body,
  Controller,
  FileTypeValidator,
  Get,
  MaxFileSizeValidator,
  Param,
  ParseFilePipe,
  Patch,
  Post,
  Put,
  Query,
  UploadedFile,
  UseGuards,
  UseInterceptors,
} from '@nestjs/common';
import { AuthGuard } from '@nestjs/passport';
import { FileInterceptor } from '@nestjs/platform-express';
import { ApiOperation } from '@nestjs/swagger';
import { Account } from 'src/account/entities/account.entity';
import { CurrentUser } from 'src/auth/decorators/current-user.decorator';
import { CheckPermissions } from 'src/auth/decorators/permissions.decorator';
import { Roles } from 'src/auth/decorators/roles.decorator';
import { PermissionsGuard } from 'src/auth/guards/permissions.guard';
import { RolesGuard } from 'src/auth/guards/roles.guard';
// import { CompanyCategoryService } from 'src/company-category/company-category.service';
// import { CompanySubCategoryService } from 'src/company-sub-category/company-sub-category.service';
import { PermissionAction, UserRole } from 'src/enum';
import { CompanyDetailsService } from './company-details.service';
import {
  BusinessDetailDto,
  CompanyDetailDto,
  PaginationDto,
  PaginationSDto,
  StatusDto,
} from './dto/company-detail.dto';

@Controller('company-details')
export class CompanyDetailsController {
  constructor(
    private readonly companyDetailsService: CompanyDetailsService,
    // private readonly companyCatService: CompanyCategoryService,
    // private readonly companySubCatService: CompanySubCategoryService,
  ) {}

  @Post()
  @ApiOperation({ summary: 'For Vendor' })
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  async create(@Body() dto: BusinessDetailDto, @CurrentUser() user: Account) {
    dto.accountId = user.id;
    return this.companyDetailsService.create(dto);
  }

  @Get('info')
  @ApiOperation({ summary: 'For Vendor' })
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  findInfo(@CurrentUser() user: Account) {
    return this.companyDetailsService.findOne(user.id);
  }

  @Get('info/:id')
  @ApiOperation({ summary: 'For User' })
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  Info(@Param('id') id: string) {
    return this.companyDetailsService.findOne(id);
  }

  @Get('list')
  @ApiOperation({ summary: 'For Users' })
  findList(@Query() dto: PaginationDto) {
    return this.companyDetailsService.findList(dto);
  }

  @Get()
  @ApiOperation({ summary: 'For Admin' })
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'company_detail'])
  findAll(@Query() dto: PaginationSDto) {
    return this.companyDetailsService.findAll(dto);
  }

  @Get(':id')
  @ApiOperation({ summary: 'For Admin' })
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.READ, 'company_detail'])
  findOne(@Param('id') id: string) {
    return this.companyDetailsService.findOne(id);
  }

  @Patch(':id')
  @ApiOperation({ summary: 'For Admin/Staff' })
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  update(@Param('id') id: string, @Body() dto: CompanyDetailDto) {
    return this.companyDetailsService.update(id, dto);
  }

  @Put('file/:id')
  @ApiOperation({ summary: 'For Vendor' })
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @UseInterceptors(FileInterceptor('file'))
  async file(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new FileTypeValidator({ fileType: '.(png|jpeg|jpg)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    return this.companyDetailsService.file(id, file.path);
  }

  @Put('logo/:id')
  @ApiOperation({ summary: 'For Vendor' })
  @UseGuards(AuthGuard('jwt'), RolesGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @UseInterceptors(FileInterceptor('file'))
  async logo(
    @Param('id') id: string,
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new FileTypeValidator({ fileType: '.(png|jpeg|jpg)' }),
          new MaxFileSizeValidator({ maxSize: 1024 * 1024 * 1 }),
        ],
      }),
    )
    file: Express.Multer.File,
  ) {
    return this.companyDetailsService.logo(id, file.path);
  }

  @Put(':id')
  @ApiOperation({ summary: 'For Admin' })
  @UseGuards(AuthGuard('jwt'), RolesGuard, PermissionsGuard)
  @Roles(UserRole.ADMIN, UserRole.SUPER_ADMIN, UserRole.STAFF)
  @CheckPermissions([PermissionAction.UPDATE, 'company_detail'])
  status(@Param('id') id: string, @Body() dto: StatusDto) {
    return this.companyDetailsService.status(id, dto);
  }
}
