import { HttpService } from '@nestjs/axios';
import { CACHE_MANAGER } from '@nestjs/cache-manager';
import {
  Inject,
  Injectable,
  NotAcceptableException,
  PreconditionFailedException,
  ServiceUnavailableException,
  UnauthorizedException,
} from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { InjectRepository } from '@nestjs/typeorm';
import * as bcrypt from 'bcrypt';
import { Cache } from 'cache-manager';
import { Account } from 'src/account/entities/account.entity';
import { DefaultStatus, LogType, OutletType, UserRole } from 'src/enum';
import { LoginHistory } from 'src/login-history/entities/login-history.entity';
import { NotifyService } from 'src/notify/notify.service';
import { UserPermission } from 'src/user-permissions/entities/user-permission.entity';
import APIFeatures from 'src/utils/apiFeatures.utils';
import { Repository } from 'typeorm';
import { MobLoginDto, PassDto } from './dto/login.dto';

@Injectable()
export class AuthService {
  constructor(
    private jwtService: JwtService,
    @InjectRepository(Account) private readonly repo: Repository<Account>,
    @InjectRepository(LoginHistory)
    private readonly logRepo: Repository<LoginHistory>,
    @InjectRepository(UserPermission)
    private readonly upRepo: Repository<UserPermission>,
    private readonly httpService: HttpService,
    @Inject(CACHE_MANAGER) private cacheManager: Cache,
    private readonly notifyService: NotifyService,
  ) {}

  async mobLogin(dto: MobLoginDto, ip: string, origin?: string) {
    if(dto.type == 'outlet'){
      const user = await this.repo.findOne({
        relations: ['outletDetail'],
        where: [
          { loginId: dto.loginId, roles: UserRole.SUB_OUTLET }
        ],
      });

      if (!user) {
        throw new UnauthorizedException('Invalid Credentials!');
      }
      const comparePassword = await bcrypt.compare(dto.password, user.password);
      if (!comparePassword) {
        if (user.wrongCount > 5) {
          this.repo
            .createQueryBuilder()
            .update()
            .set({ status: DefaultStatus.SUSPENDED })
            .where('id = :id', { id: user.id })
            .execute();
          throw new PreconditionFailedException(
            'Account suspended please contact to admin!',
          );
        }
        this.repo
          .createQueryBuilder()
          .update()
          .set({
            wrongCount: () => 'wrongCount + ' + 1,
          })
          .where('id = :id', { id: user.id })
          .execute();
  
        throw new UnauthorizedException('Invalid Credentials');
      }
  
      if (user.outletDetail[0].type === OutletType.IN_LOCATION) {
        const distance = await this.calculateDistance(
          user.outletDetail[0].latitude,
          user.outletDetail[0].longitude,
          dto.latitude,
          dto.longitude,
        );
        if (distance > 100) {
          this.notifyService.outOfLocationNotify(dto.fcm, user.id);
          throw new NotAcceptableException('Out of place login!');
        }
      }
      
  
      if (user.status !== DefaultStatus.ACTIVE) {
        this.notifyService.loginBlockedNotify(dto.fcm, user.id);
        throw new ServiceUnavailableException(
          'Account ' + user.status + '. Please contact to admin.',
        );
      }
  
      this.notifyService.loginNotify(dto.fcm, user.id);
      
  
      const deviceObj = Object.assign(user, {
        deviceId: dto.deviceId,
        fcm: dto.fcm,
        wrongCount: user.wrongCount +  1,
      });
      this.repo.save(deviceObj);
  
      const obj = Object.create({
        ip: ip,
        origin: origin,
        type: LogType.LOGIN,
        accountId: user.id,
        latitude: dto.latitude,
        longitude: dto.longitude,
        deviceId: dto.deviceId,
      });
      await this.logRepo.save(obj);
  
      const token = await APIFeatures.assignJwtToken(
        user.id,
        dto.deviceId,
        this.jwtService,
      );
      return { token, type: user.roles, pass: user.passwordShow };
    } else {
      const user = await this.repo.findOne({
        relations: ['outletDetail'],
        where: [
          { loginId: dto.loginId, roles: UserRole.DELIVERY_BOY }
        ],
      });

      if (!user) {
        throw new UnauthorizedException('Invalid Credentials!');
      }
      const comparePassword = await bcrypt.compare(dto.password, user.password);
      if (!comparePassword) {
        if (user.wrongCount > 5) {
          this.repo
            .createQueryBuilder()
            .update()
            .set({ status: DefaultStatus.SUSPENDED })
            .where('id = :id', { id: user.id })
            .execute();
          throw new PreconditionFailedException(
            'Account suspended please contact to admin!',
          );
        }
        this.repo
          .createQueryBuilder()
          .update()
          .set({
            wrongCount: () => 'wrongCount + ' + 1,
          })
          .where('id = :id', { id: user.id })
          .execute();
  
        throw new UnauthorizedException('Invalid Credentials');
      }
  
      if (user.status !== DefaultStatus.ACTIVE) {
        this.notifyService.loginBlockedNotify(dto.fcm, user.id);
        throw new ServiceUnavailableException(
          'Account ' + user.status + '. Please contact to admin.',
        );
      }
  
      this.notifyService.loginDeliveryBoyNotify(dto.fcm, user.id);

      const deviceObj = Object.assign(user, {
        deviceId: dto.deviceId,
        fcm: dto.fcm,
        wrongCount: user.wrongCount +  1,
      });
      this.repo.save(deviceObj);
  
      const obj = Object.create({
        ip: ip,
        origin: origin,
        type: LogType.LOGIN,
        accountId: user.id,
        latitude: dto.latitude,
        longitude: dto.longitude,
        deviceId: dto.deviceId,
      });
      await this.logRepo.save(obj);
  
      const token = await APIFeatures.assignJwtToken(
        user.id,
        dto.deviceId,
        this.jwtService,
      );
      return { token, type: user.roles, pass: user.passwordShow };
    }
  }

  async signIn(loginId: string, password: string, ip: string, origin?: string) {
    const admin = await this.getUserDetails(loginId);
    const comparePassword = await bcrypt.compare(password, admin.password);
    if (!comparePassword) {
      throw new UnauthorizedException('Invalid Credentials');
    }

    const token = await APIFeatures.assignJwtToken(
      admin.id,
      '1',
      this.jwtService,
    );
    const obj = Object.create({
      ip: ip,
      origin: origin,
      type: LogType.LOGIN,
      accountId: admin.id,
    });
    await this.logRepo.save(obj);

    return { token, admin };
  }

  async logout(accountId: string, browser: string, ip: string) {
    const obj = Object.create({
      ip: ip,
      type: LogType.LOGOUT,
      accountId: accountId,
    });
    return this.logRepo.save(obj);
  }

  validate(id: string, deviceId: string) {
    if (deviceId == '1') {
      return this.getUserDetails(id);
    } else {
      return this.repo
        .createQueryBuilder('account')
        .leftJoinAndSelect('account.outletDetail', 'outletDetail')
        .leftJoinAndSelect('outletDetail.outletBranch', 'outletBranch')
        .select([
          'account.id',
          'account.password',
          'account.roles',
          'account.status',
          'account.createdBy',
          'account.deviceId',
          'account.fcm',
          'account.settingId',

          'outletDetail.id',
          'outletBranch.id',
        ])
        .andWhere('account.id = :id AND account.deviceId = :deviceId', {
          id: id,
          deviceId: deviceId,
        })
        .getOne();
    }
  }

  findPermission(accountId: string) {
    return this.getPermissions(accountId);
  }

  private getPermissions = async (accountId: string): Promise<any> => {
    let result = await this.cacheManager.get('userPermission' + accountId);
    if (!result) {
      result = await this.upRepo.find({
        relations: ['permission', 'menu'],
        where: { accountId, status: true },
      });
      this.cacheManager.set(
        'userPermission' + accountId,
        result,
        7 * 24 * 60 * 60 * 1000,
      );
    }
    return result;
  };

  private getUserDetails = async (id: string): Promise<any> => {
    return this.repo
      .createQueryBuilder('account')
      .leftJoinAndSelect('account.outletDetail', 'outletDetail')
      .select([
        'account.id',
        'account.password',
        'account.roles',
        'account.status',
        'account.createdBy',
        'account.deviceId',
        'account.fcm',
        'account.settingId',

        'outletDetail.id',
      ])
      .andWhere('account.id = :id OR account.loginId = :loginId', {
        id: id,
        loginId: id,
      })
      .getOne();
  };

  calculateDistance(lat1, lon1, lat2, lon2) {
    const earthRadius = 6371e3; // Earth's radius in meters (mean value)

    // Convert latitude and longitude from degrees to radians
    const lat1Rad = lat1 * (Math.PI / 180);
    const lon1Rad = lon1 * (Math.PI / 180);
    const lat2Rad = lat2 * (Math.PI / 180);
    const lon2Rad = lon2 * (Math.PI / 180);

    // Calculate the differences between the latitudes and longitudes
    const deltaLat = lat2Rad - lat1Rad;
    const deltaLon = lon2Rad - lon1Rad;

    // Haversine formula
    const a =
      Math.sin(deltaLat / 2) * Math.sin(deltaLat / 2) +
      Math.cos(lat1Rad) *
        Math.cos(lat2Rad) *
        Math.sin(deltaLon / 2) *
        Math.sin(deltaLon / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

    // Calculate the distance in meters
    const distance = earthRadius * c;

    return distance;
  }

  async find(dto: PassDto, password) {
    const comparePassword = await bcrypt.compare(dto.password, password);
    if (!comparePassword) {
      return {"result": {
        "message": false,
        "statusCode": 401
      }};
    } else {
      return {"result": {
        "message": true,
        "statusCode": 200
      }};
    }
  }
}
